#!/usr/bin/env python3
"""
# ************************************************************************
# *                                                                      *
# *   Routine for converting VLBI schedule file from vex to snap format. *
# *   The routine conforms with vex 1.5b format. It supports keys        *
# *   start_offset and stop_offset in $SCHED section of a vex file.      *
# *   The routine has an option to check for vex file consistency. In    *
# *   checking mode it computes the idle time after completion of the    *
# *   previous scan and start of the new scan. Negative values means     *
# *   an antenna will be not on source in the scheduled time.            *
# *                                                                      *
# *   vex_to_snap.py honors station and mode specific preob and postob   *
# *   procedures specified in in the PROCEDURE section. It inserts calls *
# *   of preob and postob procedures in th snap file. If a given station *
# *   does not do anything during proob and/or postob, empty procedures  *
# *   should be defined in the procedure files.                          *
# *                                                                      *
# *   NB: vex_to_snap.py does not generate procedure files.              *
# *                                                                      *
# *   Required packages: ners ( http://earthrotation.net/ners )          *
# *                                                                      *
# *   Copyright (c) 1975-2025 United States Government as represented by *
# *   the Administrator of the National Aeronautics and Space            *
# *   Administration. All Rights Reserved.                               *
# *   License: NASA Open Source Software Agreement (NOSA).               *
# *                                                                      *
# * ### 25-JUN-2018 vex_to_snap.py v 2.3 (d)  L. Petrov 22-JUL-2023 ###  *
# *                                                                      *
# ************************************************************************
"""
import argparse, signal, sys, os, pwd, math
from   datetime import datetime, timedelta, tzinfo
from   sur_sked_config import * # Import sur_sked confuguration
import ners, vex_lib
from   vex_lib import *

vts__label   = "vex_to_snap.py"
vts__version = "2023.07.22"

supported_modes = [ "mk5", "mk5_ext", "mk6_vo", "mk6_ext", "norec" ]

#
# ==============================================================================
#
def gen_vex_to_snap ( vex, sta_name, filout, check_schedule, gain_macros_file, mode ):
    """
    Procedure gen_vex_to_snap transforms vex file to snap_file for the 
    specified station
    """
    if ( not sta_name in vex.sta ):
         print ( "Station %s does not participate in experiment %s" % \
                 ( sta_name, vex.exper_name ) )
         print ( "List of participating stations: %s" % sorted(list(vex.sta.keys())) )
         exit  ( 1 )

#
# --- Open output file
#
    if ( filout == "-" ):
         f = sys.stdout
    else:
         f = open(filout,'w',encoding="latin")

    gain_macros = {}
    mcr_name = None
    if ( gain_macros_file ):
         with open(gain_macros_file,encoding="latin") as m:
              mcr = m.read().splitlines()
         m.close()

         for line in mcr:
             if ( line[0:1] == "#" ): continue
             if ( line.split()[0] == "define" ):
                  mcr_name = line.split()[1]
                  gain_macros[mcr_name] = []
             elif ( line.split()[0] == "enddef" ):
                  mcr_name = None
             else:
                  if ( mcr_name ):
                       gain_macros[mcr_name].append( line.lstrip() )

#
# --- Initialize data struct ue for vex check
#
    che = []
#
# --- Generate the header comment section of the output snap-file
#
    print ( '" VLBI experiment schedule in snap format', file=f ) 
    print ( '" ', file=f )
    print ( '" Generated with    %s version of %s' % ( vts__label, vts__version ), file=f )
    print ( '" Generated using   %s' % vex.filename, file=f )
    print ( '" Generated via     %s' % str(" ".join(sys.argv)), file=f )
    print ( '" Generated on      %s' % \
                datetime.strftime( datetime.now(), "%Y.%m.%d_%H:%M:%S.%f")[:-5], file=f )
    print ( '" Generated by      %s' % pwd.getpwuid(os.getuid())[4], file=f )
    print ( '" ', file=f )
    print ( '" %-9s %d  %8s  %s %s' % ( vex.exper_name, \
                                         vex.exper_utc_start.year, \
                                         vex.sta[sta_name]["ant_name"], \
                                         vex.sta[sta_name]["ant_name"][0:1], \
                                         sta_name \
                                       ), file=f )

    print ( '" %s               %8s  %s' % ( vex.sta[sta_name]["ant_name"][0:1], \
                                             vex.sta[sta_name]["ant_name"], \
                                             vex.sta[sta_name]["mount"][0].upper() + \
                                             vex.sta[sta_name]["mount"][1].upper() ), file=f ) 
    print ( '" %s              %8s  %12.3f  %12.3f  %12.3f' % ( sta_name, \
                                              vex.sta[sta_name]["ant_name"], \
                                              vex.sta[sta_name]["coo"][0], \
                                              vex.sta[sta_name]["coo"][1], \
                                              vex.sta[sta_name]["coo"][2] ), file=f ) 
    print ( '"', file=f ) 
    print ( '" Experiment_code:          %s' % vex.exper_name, file=f ) 
    print ( '" Experiment_description:   %s' % vex.exper_desc, file=f ) 
    print ( '" Station:                  %-8s    %2s' % \
                                        ( vex.sta[sta_name]["ant_name"], sta_name ), file=f  )
    if ( hasattr( vex, "revision") ):
         print ( '" Schedule_revision:        %s' % vex.revision, file=f ) 
    if ( hasattr( vex, "contact_name") ):
         print ( '" Scheduler_name:           %s' % vex.contact_name,  file=f ) 
    if ( hasattr( vex, "contact_email") ):
         print ( '" Scheduler_email:          %s' % vex.contact_email, file=f ) 
    print ( '" Scheduler_phone:          n/a', file=f  )
    print ( '" Observer_phone:           n/a', file=f  )
    print ( '" UTC_experiment_dates:     %s, %s' % \
             ( datetime.strftime( vex.exper_utc_start, "%Y.%m.%d_%H:%M:%S.%f")[:-5], \
               datetime.strftime( vex.exper_utc_stop,  "%Y.%m.%d_%H:%M:%S.%f")[:-5]  \
             ), file=f  )
    print ( '" ', file=f )
    print ( '" Station parameters used for generating this schedule:', file=f )
    print ( '" ', file=f )
    print ( '" Last_update:              %-8s    n/a' % vex.sta[sta_name]["ant_name"], file=f )
    print ( '" Coordinates:              %-8s  %12.3f  %12.3f  %12.3f' % ( \
                                vex.sta[sta_name]["ant_name"], \
                                vex.sta[sta_name]["coo"][0], \
                                vex.sta[sta_name]["coo"][1], \
                                vex.sta[sta_name]["coo"][2] ), file=f ) 
    print ( '" Mount:                    %-8s   %s' % ( \
                                vex.sta[sta_name]["ant_name"], \
                                vex.sta[sta_name]["mount"][0].upper() + \
                                vex.sta[sta_name]["mount"][1].upper() ), file=f ) 
    print ( '" 1st_axis_range:           %-8s  %6.1f %6.1f %6.1f %6.1f  deg  Axis: %s' % \
                         ( vex.sta[sta_name]["ant_name"],            \
                           vex.sta[sta_name]["az_lim"][0]*180.0/ners.NERS__PI,     \
                           vex.sta[sta_name]["az_lim"][1]*180.0/ners.NERS__PI,     \
                           vex.sta[sta_name]["az_lim"][2]*180.0/ners.NERS__PI,     \
                           vex.sta[sta_name]["az_lim"][3]*180.0/ners.NERS__PI,     \
                           vex.sta[sta_name]["mount"][0] ), file=f \
                         )
    print ( '" 2nd_axis_range:           %-8s  %6.1f %6.1f                deg  Axis: %s' % \
                         ( vex.sta[sta_name]["ant_name"],            \
                           vex.sta[sta_name]["el_lim"][0]*180.0/ners.NERS__PI,     \
                           vex.sta[sta_name]["el_lim"][1]*180.0/ners.NERS__PI,     \
                           vex.sta[sta_name]["mount"][1] ), file=f \
                         )
    print ( '" 1st_axis_slewing_rate:    %-8s  %6.2f    deg/sec                 Axis: %s' % \
                         ( vex.sta[sta_name]["ant_name"],            \
                           vex.sta[sta_name]["slew_rate"][0]*180.0/ners.NERS__PI,  \
                           vex.sta[sta_name]["mount"][0] ), file=f \
                         )
    print ( '" 2nd_axis_slewing_rate:    %-8s  %6.2f    deg/sec                 Axis: %s' % \
                         ( vex.sta[sta_name]["ant_name"],            \
                           vex.sta[sta_name]["slew_rate"][1]*180.0/ners.NERS__PI,  \
                           vex.sta[sta_name]["mount"][1] ), file=f \
                         )
    print ( '" 1st_axis_slewing_accl:    %-8s  %6.2f    deg/sec^2               Axis: %s' % \
                         ( vex.sta[sta_name]["ant_name"],            \
                           vex.sta[sta_name]["slew_accel"][0]*180.0/ners.NERS__PI, \
                           vex.sta[sta_name]["mount"][0] ), file=f \
                         )
    print ( '" 2nd_axis_slewing_accl:    %-8s  %6.2f    deg/sec^2               Axis: %s' % \
                         ( vex.sta[sta_name]["ant_name"],            \
                           vex.sta[sta_name]["slew_accel"][1]*180.0/ners.NERS__PI, \
                           vex.sta[sta_name]["mount"][1] ), file=f \
                         )
    print ( '" 1st_axis_settle_time:     %-8s  %6.2f    sec                     Axis: %s' % \
                         ( vex.sta[sta_name]["ant_name"],            \
                           vex.sta[sta_name]["slew_tsettle"][0],     \
                           vex.sta[sta_name]["mount"][0] ), file=f   \
                         )
    print ( '" 2nd_axis_settle_time:     %-8s  %6.2f    sec                     Axis: %s' % \
                         ( vex.sta[sta_name]["ant_name"],             \
                           vex.sta[sta_name]["slew_tsettle"][1],      \
                           vex.sta[sta_name]["mount"][1]  ), file=f   \
                         )
    if ( "hor_mask_az" in vex.sta[sta_name].keys() ):
          str_hor_mask = ""
          for azim in vex.sta[sta_name]["hor_mask_az"]:
              str_hor_mask = str_hor_mask + "%6.2f " % azim
          print ( '" AZ_hor_mask:              %s  %s ' % \
                  ( vex.sta[sta_name]["ant_name"], str_hor_mask ), file=f )
    else:
          print ( '" Horizontal mask is not defined', file=f )

    if ( "hor_mask_el" in vex.sta[sta_name].keys() ):
          str_hor_mask = ""
          for elev in vex.sta[sta_name]["hor_mask_el"]:
              str_hor_mask = str_hor_mask + "%6.2f " % elev
          print ( '" EL_hor_mask:              %s  %s ' % \
                  ( vex.sta[sta_name]["ant_name"], str_hor_mask ), file=f )

    if ( not "proc_names" in vex.sta[sta_name].keys() ):
         print ( "\nERROR: No procedures were defined for station %s" % sta_name )
         exit  ( 1 )

    for proc_name in vex.sta[sta_name]["proc_names"]:
        if ( not "preob_proc_dur" in vex.proc[proc_name].keys() ):
             print ( "\nERROR: Procedure preob_proc is not defined for station %s" % sta_name )
             exit  ( 1 )
        if ( not "postob_proc_dur" in vex.proc[proc_name].keys() ):
             print ( "\nERROR: Procedure postob_proc is not defined for station %s" % sta_name )
             exit  ( 1 )

#        print ( '" Preob_proc_duration:      %-8s  %6.2f    sec  hds: %s' %  \
#                           ( vex.sta[sta_name]["ant_name"],                  \
#                             vex.proc[proc_name]["preob_proc_dur"],          \
#                             vex.proc[proc_name]["hds"]                      \
#                           ), file=f                                         \
#              )
#        print ( '" Postob_proc_duration:     %-8s  %6.2f    sec  hds: %s' % \
#                            ( vex.sta[sta_name]["ant_name"],                 \
#                              vex.proc[proc_name]["postob_proc_dur"],        \
#                              vex.proc[proc_name]["hds"]                     \
#                            ), file=f                                        \
#              )
    print ( '" Recorder:                 %-8s  %s' % \
                         ( vex.sta[sta_name]["ant_name"],             \
                           vex.sta[sta_name]["das_name"] ), file=f    \
                         )
    if ( "backend" in vex.sta[sta_name].keys() ):
          print ( '" Backend:                  %-8s  %s' % \
                         ( vex.sta[sta_name]["ant_name"],             \
                           vex.sta[sta_name]["backend"] ), file=f     \
                         )
    else:
          print ( '" Backend:                  %-8s  %s' % \
                         ( vex.sta[sta_name]["ant_name"], \
                           "unknown" ), file=f    \
                         )
    print ( '" Recording_rate:           %-8s  %6.3f    Gbps' %        \
                         ( vex.sta[sta_name]["ant_name"],             \
                           vex.sta[sta_name]["rec_rate"]/1000.0 ), file=f  \
                         )
    print ( '" ', file=f )

#
# --- Get sorted scan list
#
    scan_list  = sorted(list(vex.scan.keys()))
#
# --- Initialization
#
    ready_disk = None
    old_az_acc = None
    old_az     = None
    az_acc     = 0.0
    old_el     = None
    old_ha     = None
    old_sector = None
    last_scan  = " "
    tot_rec_time = 0
#
# --- Process vex file for a given station, scan by scan
#
    if ( gain_macros ):
         print ( '"Presess commands', file=f ) 
         for line in gain_macros["presess"]:
             print ( line, file=f ) 
         print ( '"', file=f ) 

    cur_mode = "Unknown"
    ind_sca = 0
    for scan_name in scan_list:
        ind_sca = ind_sca + 1
        if ( sta_name in list(vex.scan[scan_name]["station"]) ):
             if ( not "mode" in vex.scan[scan_name] ):
                  print ( "Mode has not been defined for the %d scan %s" % \
                          (ind_sca, scan_name ) )
                  exit ( 1 )
             mode_name = None
             for proc_name_test in vex.sta[sta_name]["proc_names"]:
                 for proc_test_setmode in vex.proc[proc_name_test]["setmode"]:
                     if ( proc_test_setmode == vex.scan[scan_name]["mode"] ):
                          mode_name = proc_test_setmode 
             if ( not mode_name ):
                  print ( "Did not find setmode procedure for mode %s in procesdure %s" % \
                          ( vex.scan[scan_name]["mode"], vex.sta[sta_name]["proc_names"] ) )
                  exit ( 1 )
#
# ---------- Print the line with scan name, experiment name, source name, and scan diration
#
             print ( "scan_name=%s,%s,%s,%d,%d" % \
                       ( scan_name, \
                         vex.exper_name, \
                         sta_name, \
                         vex.scan[scan_name]["station"][sta_name]["stop_offset"]         \
                             - vex.scan[scan_name]["station"][sta_name]["start_offset"], \
                         vex.scan[scan_name]["station"][sta_name]["stop_offset"]         \
                             - vex.scan[scan_name]["station"][sta_name]["start_offset"]  \
                       ), file=f )
             sou_ra_str  = ners.rad_to_hms ( vex.sou[vex.scan[scan_name]["source"]]["ra"], 4 ) 
             sou_dec_str = ners.rad_to_dms ( vex.sou[vex.scan[scan_name]["source"]]["dec"], 3, "+"  )

             sector = vex.scan[scan_name]["station"][sta_name]["sector"][1:]
             if ( sector == "n" ): sector = "neutral"
#
# ---------- Print a line with source name, source coordinates, and the azimuth sector ID
#
             print ( "source=%s,%s,%s,2000.0,%s" % \
                       ( vex.scan[scan_name]["source"], \
                         sou_ra_str.replace(":",""), \
                         sou_dec_str.replace(":",""), \
                         sector, \
                       ), file=f )
#
# ---------- Compute azimuth, elevation, and hour angle 
# ---------- at the beginning of the preobs procedure using NERS package
#
             preobs_start_utc_time = vex.scan[scan_name]["start_utc_time"] + \
                                     timedelta ( seconds=vex.scan[scan_name]["station"][sta_name]["start_offset"] ) - \
                                     timedelta ( seconds=vex.proc[proc_name]["preob_proc_dur"] )
             (az, el, ha) = ners.get_azelha ( preobs_start_utc_time, \
                                        vex.sta[sta_name]["coo"], \
                                        vex.sou[vex.scan[scan_name]["source"]]["ra"], \
                                        vex.sou[vex.scan[scan_name]["source"]]["dec"], \
                                        "radio", ners.NERS__UTC )
#
# ---------- Modify azimuth depending on the azimuthal sector
#
             if ( sector == "ccw" ): 
                  if ( az >= vex.sta[sta_name]["az_lim"][0] and \
                       az <= vex.sta[sta_name]["az_lim"][1]     ):
                       az_acc = az
                  elif ( az > vex.sta[sta_name]["az_lim"][2] ):
                       az_acc = az - 2*ners.NERS__PI 
                  else:
                       az_acc = az

             if ( sector == "neutral" ): 
                  if ( az >= vex.sta[sta_name]["az_lim"][1] and \
                       az <= vex.sta[sta_name]["az_lim"][2]     ):
                       az_acc = az
                  elif ( az >= vex.sta[sta_name]["az_lim"][0] and \
                         az <= vex.sta[sta_name]["az_lim"][1]     ):
                       az_acc = az + 2*ners.NERS__PI 
                  elif ( az >= vex.sta[sta_name]["az_lim"][2] and \
                         az <= vex.sta[sta_name]["az_lim"][3]     ):
                       az_acc = az - 2*ners.NERS__PI 

             if ( sector == "cw" ): 
                  if ( az >= vex.sta[sta_name]["az_lim"][2] and \
                       az <= vex.sta[sta_name]["az_lim"][3]     ):
                       az_acc = az
                  elif ( az < vex.sta[sta_name]["az_lim"][2] ):
                       az_acc = az + 2*ners.NERS__PI 
                  else:
                       az_acc = az

             if ( not old_az ):
#
# --------------- The is th fist scan. Print comment: new azimuth, elevation, hour angle
#
                  print ( '" New az/el/ha: %6.2f,  %6.2f,  %6.2f' % \
                            ( az*180.0/ners.NERS__PI, \
                              el*180.0/ners.NERS__PI, \
                              ha*180.0/ners.NERS__PI ), file=f )
             else: 
#
# --------------- This is not the first scan.
# --------------- Compute difference in azimuth and elevation
#
                  delta_az = abs(az_acc - old_az_acc)
                  delta_el = abs(el - old_el)
                  if ( vex.sta[sta_name]["slew_accel"][0] == 0.0 ): 
                       print ( "vex+_to_snap.pt ERROR antenna acceleration over the 1st axis of station %s is zero" %  sta_name )
                       exit  ( 1 )
                  if ( vex.sta[sta_name]["slew_accel"][1] == 0.0 ):
                       print ( "vex+_to_snap.pt ERROR antenna acceleration over the 2nd axis of station %s is zero" %  sta_name )
                       exit  ( 1 )
#
# --------------- Compute slewing time over azimuth
#
                  if ( delta_az > vex.sta[sta_name]["slew_rate"][0]**2/vex.sta[sta_name]["slew_accel"][0] ):
                       slew_az = delta_az/vex.sta[sta_name]["slew_rate"][0] + \
                                 vex.sta[sta_name]["slew_rate"][0]/vex.sta[sta_name]["slew_accel"][0]
                       flag_acc = "f"
                  else:
                       slew_az = 2.0*math.sqrt(delta_az/vex.sta[sta_name]["slew_accel"][0])
                       flag_acc = "s"

#
# --------------- Compute slewing time over elevation
#
                  if ( delta_el > vex.sta[sta_name]["slew_rate"][1]**2/vex.sta[sta_name]["slew_accel"][1] ):
                       slew_el = delta_el/vex.sta[sta_name]["slew_rate"][1] + \
                                 vex.sta[sta_name]["slew_rate"][1]/vex.sta[sta_name]["slew_accel"][1]
                       flag_acc = "f"
                  else:
                       slew_el = 2.0*math.sqrt(delta_el/vex.sta[sta_name]["slew_accel"][1])
                       flag_acc = "s"

#
# --------------- Compute the toal slewing time
#
                  slew_time = max ( slew_az + vex.sta[sta_name]["slew_tsettle"][0], \
                                    slew_el + vex.sta[sta_name]["slew_tsettle"][1]  )
                  if ( slew_el > slew_az ):
                       slew_mode = 'E' + flag_acc
                  else:
                       slew_mode = 'A' + flag_acc

#
# --------------- Print comment line with new and old azimuth, elevation, and hour angle, 
# --------------- as well as the total slew time
#
                  print ( '" New az/el/ha:  %6.2f, %6.2f, %7.2f  Old az/el/ha:  %6.2f, %6.2f, %7.2f  Slewing time: %6.1f sec' % \
                            ( az*180.0/ners.NERS__PI, \
                              el*180.0/ners.NERS__PI, \
                              ha*180.0/ners.NERS__PI,
                              old_az*180.0/ners.NERS__PI, \
                              old_el*180.0/ners.NERS__PI, \
                              old_ha*180.0/ners.NERS__PI, \
                              slew_time, \
                             ), file=f )
#
# --------------- Print comment line with path over azimuth and elevation as well as 
# --------------- slewing time over azimuth and elevation, settle time, and slewing mode
#
                  print ( '" D_az/D_el:     %6.2f, %6.2f  Slew_az/Slew_el: %5.1f, %5.1f  Settle_time_az/Settle_time_el: %5.1f, %5.1f  Slew_mode: %s' % \
                             ( delta_az*180.0/ners.NERS__PI, \
                               delta_el*180.0/ners.NERS__PI, \
                               slew_az, \
                               slew_el, \
                               vex.sta[sta_name]["slew_tsettle"][0], \
                               vex.sta[sta_name]["slew_tsettle"][1], \
                               slew_mode ), file=f )
#
# --------------- Extract the nominal slew time computed by the scheduling program
# --------------- and compute the idle time before preob procedure
#
                  nomimal_slew_time = (preobs_start_utc_time - old_postob_stop_utc_time).seconds + \
                                      (preobs_start_utc_time - old_postob_stop_utc_time).days*86400.0
                  idle_after_slew = nomimal_slew_time - slew_time
                  time_to_stop_slewing = postob_stop_utc_time + timedelta ( seconds=slew_time )
#
# --------------- Print a comment line about idle time after slewing and
# --------------- before preobs
#
                  print ( '" Slew ends at %s Antenna is idle after slewing for  %7.1f sec' % \
                          ( datetime.strftime( time_to_stop_slewing, "%Y.%j.%H:%M:%S.%f")[0:20], \
                            idle_after_slew ), file=f )
#
# --------------- Generate a line for schedule slewing analysis
#
                  che.append ( "%s  Scan %16s Slew time: %5.1f  Idle : %1s %8.1f" % \
                                   ( sta_name, scan_name, slew_time, last_scan, idle_after_slew ) )

             if ( vex.scan[scan_name]["intent"] != [] ):
                  for intent in vex.scan[scan_name]["intent"]:
                      print ( '" scan: %-16s  intent: %s' % (scan_name, intent), file=f )

             if ( not gain_macros ):
#
# --------------- Print Mark-5 specific commands
#
                  if ( not mode == "mk5_ext" ):
                       if ( "MARK5" in vex.sta[sta_name]["das_name"] or 
                            "mark5" in vex.sta[sta_name]["das_name"]    ):
                             if ( ready_disk ):
                                  print ( "checkmk5", file=f ) 
                                  print ( "ready_disk", file=f ) 
                                  ready_disk = "ok"
#                            print ( "setup01", file=f  )

             if ( vex.scan[scan_name]["mode"] != cur_mode ):
                  if ( mode == "mk5_ext" or mode == "mk6_ext" ): 
                       print ( "setmode_" + vex.scan[scan_name]["mode"], file=f )
                  cur_mode = vex.scan[scan_name]["mode"] 

             if ( len(vex.mode[vex.scan[scan_name]["mode"]]["freq"].keys()) > 1 ):
                  ind_freq_mode = None
                  freq_counter = -1
                  for freq_mode in vex.mode[vex.scan[scan_name]["mode"]]["freq"]: 
                      freq_counter = freq_counter + 1
                      if ( sta_name in vex.mode[vex.scan[scan_name]["mode"]]["freq"][freq_mode]["station"]  ):
                           ind_freq_mode = freq_counter
                  if ( ind_freq_mode == None ):
                       print ( "Did not found frequency mode for station ", sta_name )
                       print ( "vex.mode  = ", vex.mode )
                       exit ( 1 )
             else:
                  ind_freq_mode = 0
                   
             freq_name = list(vex.mode[vex.scan[scan_name]["mode"]]["freq"].keys())[ind_freq_mode]
             if ( not freq_name in vex.freq.keys() ):
                  print ( "Malformed vex file: unknown frequency name %s was found in $FREQ section" % \
                           freq_name )
                  exit  ( 1 )
           
             bit_rate_mbps = int(vex.freq[freq_name]["aggregated_bits_rate"])
#
# ---------- Print start of preob procedure
#
             if ( vex.sta[sta_name]["das_name"] == "mark6_recorder"    or \
                  vex.sta[sta_name]["das_name"] == "flexbuff_recorder" or \
                  vex.sta[sta_name]["das_name"] == "mark5_recorder"    or \
                  vex.sta[sta_name]["das_name"] == "mark5b_recorder"      ):
                  if ( mode == "mk6_vo" and ind_sca > 1 ): 
                       prerec_start_utc_time = preobs_start_utc_time - timedelta ( seconds=4.0 )
                       print ( "!%s" % datetime.strftime( prerec_start_utc_time, "%Y.%j.%H:%M:%S"), file=f )
                       print ( "mk6=rtime?%d;" % bit_rate_mbps, file=f )
                       print ( "checkmk6",        file=f )
                  if ( mode == "mk5_ext" or mode == "mk6_ext" ): 
                       print ( vex.proc[proc_name]["setscan_proc_name"], file=f )
                  if ( mode != "mk5_ext" and mode != "mk6_ext" ): 
                       print ( "setupbb", file=f )

             if ( vex.sta[sta_name]["das_name"] == "mark6_recorder" ):
                  print ( "mk6=record=%s:%d:%d:%s:%s:%s;" % \
                          ( datetime.strftime( vex.scan[scan_name]["start_utc_time"], "%Yy%jd%Hh%Mm%Ss"), 
                              vex.scan[scan_name]["station"][sta_name]["stop_offset"]   \
                            - vex.scan[scan_name]["station"][sta_name]["start_offset"], \
                              vex.scan[scan_name]["station"][sta_name]["stop_offset"]   \
                            - vex.scan[scan_name]["station"][sta_name]["start_offset"], \
                            scan_name, \
                            vex.exper_name, \
                            sta_name ), file=f )

             print ( "!%s" % datetime.strftime( preobs_start_utc_time, "%Y.%j.%H:%M:%S"), file=f )
#
# ---------- Compute record time for this scan and augment 
# ---------- the total record time accummulator
#
             record_start_utc_time = preobs_start_utc_time + \
                                     timedelta ( seconds=vex.proc[proc_name]["preob_proc_dur"] )
             record_stop_utc_time = record_start_utc_time + \
                                        timedelta ( seconds=vex.scan[scan_name]["station"][sta_name]["stop_offset"]  ) \
                                      - timedelta ( seconds=vex.scan[scan_name]["station"][sta_name]["start_offset"] )
             tot_rec_time = tot_rec_time \
                              + vex.scan[scan_name]["station"][sta_name]["stop_offset"]  \
                              - vex.scan[scan_name]["station"][sta_name]["start_offset"] 

             postob_stop_utc_time = record_stop_utc_time + \
                                    timedelta ( seconds=vex.proc[proc_name]["postob_proc_dur"] )
             if ( not gain_macros ):
                  if ( mode == "mk5_ext" or mode == "mk6_ext" ): 
                       print ( vex.proc[proc_name]["preob_proc_name"], file=f )
                  if ( mode != "mk5_ext" and mode != "mk6_ext" ): 
                        print ( "preob", file=f )
                  if ( mode == "mk6_vo" ):
                       print ( "mk6=rtime?%d;" % bit_rate_mbps, file=f )

# --------------- Print start of recording time tag
#
                  print ( "!%s" % datetime.strftime( record_start_utc_time, "%Y.%j.%H:%M:%S"), file=f )
                  if ( mode != "mk5_ext" and mode != "mk6_ext" ): 
                       if ( "MARK5" in vex.sta[sta_name]["das_name"] or 
                            "mark5" in vex.sta[sta_name]["das_name"]    ):
                            print ( "disk_pos", file=f )
                            print ( "disk_record=on", file=f )
                            print ( "disk_record", file=f )

                       print ( "data_valid=on", file=f )

                  if ( mode == "mk6_vo" ):
                       print ( "mk6=rtime?%d;" % bit_rate_mbps, file=f )
                  if ( mode == "mk5_ext" or mode == "mk6_ext" ): 
                       print ( vex.proc[proc_name]["midob_proc_name"], file=f )
                  if ( mode != "mk5_ext" and mode != "mk6_ext" ): 
                       print ( "midob", file=f )
#
# --------------- Print end of recording time tag
#
                  print ( "!%s" % datetime.strftime( record_stop_utc_time, "%Y.%j.%H:%M:%S"), file=f )
                  if ( mode != "mk5_ext" and mode != "mk6_ext" ): 
                       print ( "data_valid=off", file=f )
                       if ( "MARK5" in vex.sta[sta_name]["das_name"] or 
                            "mark5" in vex.sta[sta_name]["das_name"]    ):
                            print ( "disk_record=off", file=f )
                            print ( "disk_pos", file=f )
                  if ( mode == "mk5_ext" or mode == "mk6_ext" ): 
                       print ( vex.proc[proc_name]["postob_proc_name"], file=f )
                  if ( mode != "mk5_ext" and mode != "mk6_ext" ): 
                       print ( "postob", file=f )
             
#
# --------------- Print a comment with time tag of the end of postob procedure
#
                  if ( ind_sca  < len(scan_list) ):
                       print ( '" %s  Start slewing to the next source' % datetime.strftime( postob_stop_utc_time, "%Y.%j.%H:%M:%S"), file=f )
                  else:
#
# -------------------- or the end of the scan
#
                       print ( '" %s  End of %s schedule' % ( datetime.strftime( postob_stop_utc_time, "%Y.%j.%H:%M:%S"), \
                                                              vex.exper_name ), file=f )
                  print ( '"',  file=f ) 
             else:
                  print ( '"Gain commands', file=f ) 
                  for line in gain_macros["gain"]:
                      print ( line, file=f ) 
      
                  print ( '"',  file=f ) 
                  print ( '" %s  Start slewing to the next source' % datetime.strftime( postob_stop_utc_time, "%Y.%j.%H:%M:%S"), file=f )
                  print ( '"',  file=f ) 

#
# ---------- Update "old" azimuths, elevations, hour angles, sector id, etc
#
             old_az     = az
             old_el     = el
             old_ha     = ha
             old_az_acc = az_acc
             old_sector = sector
             old_postob_stop_utc_time = postob_stop_utc_time
             last_scan = " "
        else:
             last_scan = "m"

#
# --- Print the total recording imne adn amount of disk space used
#
    if ( not gain_macros ):
         print ( '" Total recording time:  %8.1f or  %11.4g bytes' % \
                ( tot_rec_time, tot_rec_time*bit_rate_mbps*1.e6/8 ), file=f )
    else:
         print ( '"Postsess commands', file=f ) 
         for line in gain_macros["postsess"]:
             print ( line, file=f ) 


    if ( mode != "mk5_ext" and mode != "mk6_ext" ): 
#
# ------ Print a mark-5 specific command
#
         if ( "MARK5" in vex.sta[sta_name]["das_name"] or 
              "mark5" in vex.sta[sta_name]["das_name"]    ):
              print ( "checkmk5", file=f )

#
# --- End of schedule line!
#
    if ( mode == "mk5_ext" or mode == "mk6_ext" ): 
         print ( vex.proc[proc_name]["postses_proc_name"], file=f )
    else:
         print ( "sched_end", file=f )

    if ( check_schedule ):
#
# ------ Print a biffer with results of schedule scheck
#
         for line in che:
             print ( line )

    f.close()
#
# ------------------------------------------------------------------------
#
def main():
    """ 
    Main program of the vex_to_snap utility
    """
#
# --- Parsing vex_to_smap arguments 
#
    parser = argparse.ArgumentParser( description=vts__label )
    parser.add_argument('--version', action='version', version=vts__version )

#
# --- General options:
#
    parser.add_argument ( "-v", "--verbosity", \
                          action="store", \
                          dest="verb", \
                          default=0, \
                          metavar="value", \
                          type=int, \
                          help="Verbosity level" )

    parser.add_argument ( "-c", "--check", \
                          action="store", \
                          dest="check", \
                          help="Check schedule consistency" )

    parser.add_argument ( "-g", "--gain", \
                          action="store", \
                          dest="gain", \
                          metavar="value", \
                          help="File with macros defintions for gain measurements" )

    parser.add_argument ( "-m", "--mode", \
                          action="store", \
                          dest="mode", \
                          metavar="value", \
                          help="Specfial mode" )

    parser.add_argument ( "-o", "--output", \
                          action="store", \
                          default="-", \
                          metavar="output", \
                          dest="output", \
                          help="Output file in snap format" )

#
# -- Define positional arguments
#
    parser.add_argument ( "vex_file", \
                          help="Input VEX file" )

    parser.add_argument ( "sta_name", \
                          help="2-letter long station name" )

#
# --- Get and parse options
#
    args = parser.parse_args()

    if ( not args.vex_file ):
         print ( "Vex file argumnet should be provided" )
         exit ( 1 )

    if ( not args.sta_name ):
         print ( "Station name argumnet should be provided" )
         exit ( 1 )

    if ( not os.path.isfile ( args.vex_file ) ):
         print ( "Cannot find vex file %s" % args.vex_file )
         exit ( 1 )

    if ( not args.mode ):
         print ( "mode should be specified. Supported modes: %s" % \
                  str(", ".join(supported_modes)) )
         exit ( 1 )
    else:
         if ( not args.mode in supported_modes ):
              print ( "mode %s not supported. Only %s are supported" % \
                      ( args.mode, str(", ".join(supported_modes)) ) )
              exit ( 1 )
#
# --- Parse vex file
#
    vex = parse_vex ( args.vex_file )
#
# --- Prints its contents for debugging depending on parameter arvs.verb
#
    print_vex ( vex, args.verb )
#
# --- Generate output snap file(s)
#
    if ( os.path.isdir(args.output) and args.sta_name.lower() == "all" ):
#
# ------ Case of "all" stations
#
         sta_list = sorted(list(vex.sta.keys()))
         for sta_name in sta_list:
             output_file = args.output + "/" + vex.exper_name + sta_name + ".snp"
             outpuf_file = output_file.replace("//","/")

             if ( not "coo" in vex.sta[sta_name].keys() ):
                  print ( "vex file %s does not contain coordinates of station %s" % \
                          ( args.vex_file, sta_name ) )
                  exit  ( 1 )

             gen_vex_to_snap ( vex, sta_name, output_file, args.check )

    elif ( not os.path.isdir(args.output) and args.sta_name.lower() == "all" ):
         print ( "Argument %s is not a directory name" % args.output )
         exit ( 1 )
    else:
#
# ------ Case: given station
#
         if ( not args.sta_name.lower() in list(vex.sta.keys()) ):
              print ( "Station %s does not participate in experiment %s" % \
                      ( args.sta_name.lower(), vex.exper_name ) )
              print ( "List of participating stations: %s" % sorted(list(vex.sta.keys())) )
              exit ( 1 )
         if ( os.path.isdir(args.output) ):
#
# ----------- Ouput name is directory. Build the output file name
#
              output_file = args.output + "/" + vex.exper_name + args.sta_name + ".snp"
              output_file = output_file.replace("//","/")
              gen_vex_to_snap ( vex, args.sta_name, output_file, args.check, args.mode )
         else:
#
# ----------- Ouptut name is file. Build the output file name
#
             if ( not "coo" in vex.sta[args.sta_name.lower()].keys() ):
                  print ( "vex file %s does not contain coordinates of station %s" % \
                          ( args.vex_file, args.sta_name.lower() ) )
                  exit  ( 1 )

             gen_vex_to_snap ( vex, args.sta_name, args.output, args.check, args.gain, args.mode  )

if __name__ == "__main__":
    try:
        vers = "%02d%02d%03d" % ( sys.version_info.major, sys.version_info.minor, sys.version_info.micro )
        if ( vers < "0302000" ): print ( "This script cannot run under Python older than 3.2. Please upgrade" ); exit ( 1 )
        main()
    except KeyboardInterrupt:
        print ( "pf.py: Interrupted" )
        exit ( 1 )
