      SUBROUTINE GVH_FROM_NUSOLVE ( GVH_NS, GVH, GVF_DB_DIR, FILDB, &
     &                              M_ENV, L_ENV, BUF_ENV, IUER )
! ************************************************************************
! *                                                                      *
! *   Routine  GVH_FROM_NUSOLVE  transforms the VLBI database in VGODSA  *
! *   format generated by nuSolve to the form that can be ingested to    *
! *   pSolve.                                                            *
! *                                                                      *
! * _________________________ Input parameters: ________________________ *
! *                                                                      *
! *     GVH_NS ( GVH__STRU ) -- Data structure which keeps internal      *
! *                             information related to the database of   *
! *                             an astro/geo VLBI experiment generated   *
! *                             by nuSolve.                              *
! *        GVH ( GVH__STRU ) -- Data structure which keeps internal      *
! *                             information related to the database of   *
! *                             an astro/geo VLBI experiment transformed *
! *                             to the form accepted by pSolve.          *
! * GVF_DB_DIR ( CHARACTER ) -- Name of the directory with binary files  *
! *                             of the VLBIO databases in GVF format.    *
! *      FILDB ( CHARACTER ) -- Output database file name.               *
! *      M_ENV ( INTEGER*4 ) -- The maximum number of GVF sections.      *
! *                                                                      *
! * _________________________ Output parameters: _______________________ *
! *                                                                      *
! *      L_ENV ( INTEGER*4 ) -- The number of GVF sections.              *
! *    BUF_ENV ( CHARACTER ) -- Array with the contents of the GVF       *
! *                             envelope.                                *
! *                                                                      *
! * ________________________ Modified parameters: ______________________ *
! *                                                                      *
! *    IUER ( INTEGER*4, OPT ) -- Universal error handler.               *
! *                           Input: switch IUER=0 -- no error messages  *
! *                                  will be generated even in the case  *
! *                                  of error. IUER=-1 -- in the case of *
! *                                  error the message will be put on    *
! *                                  stdout.                             *
! *                           Output: 0 in the case of successful        *
! *                                   completion and non-zero in the     *
! *                                   case of error.                     *
! *                                                                      *
! *   Copyright (c) 1975-2025 United States Government as represented by *
! *   the Administrator of the National Aeronautics and Space            *
! *   Administration. All Rights Reserved.                               *
! *   License: NASA Open Source Software Agreement (NOSA).               *
! *                                                                      *
! * ### 23-JUN-2019  GVH_FROM_NUSOLVE v1.5 (d) L. Petrov 23-AUG-2023 ### *
! *                                                                      *
! ************************************************************************
      IMPLICIT   NONE 
#ifdef GVH_STANDALONE
      INCLUDE   'gvh_solve.i'
#else
      INCLUDE   'solve.i'
#endif
      INCLUDE   'astro_constants.i'
      INCLUDE   'gvh.i'
      INCLUDE   'gvf_db.i'
      INCLUDE   'ners.i'
      INCLUDE   'ners_local.i'
      TYPE     ( GVH__STRU ) :: GVH, GVH_NS
      TYPE     ( CAL__INFO__TYPE ) :: CAL_INFO(GVF_DB__MCAL)
      TYPE     ( NERS__TYPE ) :: NERS
      INTEGER*4  M_ENV, L_ENV, IUER
      CHARACTER  GVF_DB_DIR*(*), FILDB*(*), BUF_ENV(M_ENV)*(*)
      INTEGER*4  M_LCD
      INTEGER*4  J1, J2, J3, J4, J5, J6, J7, J8, J9, J10, J11, J12, J13, J14, &
     &           J15, J16, J17, J18, J19, J20, J21, J22, J23, J24, J25, J26, J27, &
     &           J28, J29, J30, J31, J32, J33, J34, J35, J36, J37, J38, J39, &
     &           J40, J41, J42, J43, J44, J45, J46, J47, J48, J49, J50, J51, &
     &           NUM_AP(2,GVF_DB__MFRQ), NUM_CHNS(2), L_LCD, IP, IP1, IP2, &
     &           KP, IR, I_CAL, L_CAL, IND_SOU, MJD_OBS(GVH__MOBS), IER
      INTEGER*2  BUSE_I2(MAX_ARC_STA*MAX_ARC_STA), DATYP_I2, SUPMET_I2, VAL_I2_ARR(8192)
      CHARACTER  C_LCD(GVH__MTOC)*8, C_STA(GVH__MSTA)*8, DESCR*1024, STR*128, &
     &           C_CAL(GVH__MTOC)*8, EXP_CODE*32, SCA_NAME*32, C_SCA(GVH__MSCA)*16, &
     &           C_SCA_SOU(GVH__MSCA)*8, C_SOU(GVH__MSOU)*8
      INTEGER*4  BAS_USE(MAX_ARC_BSL), STA_USE(GVH__MSTA), BSCL_EST(MAX_ARC_BSL)
      INTEGER*4  CLASS, TYP, DIMS(2), LEN_REC, LEN_DATA, SEG_IND, NUM_FIELDS, &
     &           NUM_BAS_RW, MLEN_COHERCOR, IND_CAB_CAL, IND_UNP_CAL, NUM_CHANS
      INTEGER*4  NUMOBS, NUMSCA, NUMSTA, NUMSOU, NUMBAND, NOBS_STA(GVH__MSTA), &
     &           OBS_TAB(3,GVH__MOBS), CAL_STS(GVF_DB__MCAL*GVH__MSTA), EDIT_STS, &
     &           TEC_STS_STA(GVH__MSTA), NUM_CLRF, SOCO_EST(2,GVH__MSOU), &
     &           SOU_USE(GVH__MSOU), IS, FRTYPFIT, L_SCA
      INTEGER*2  IND_CHN1(GVF_DB__MFRQ), IND_CHN2(GVF_DB__MFRQ), NUSEDCHN(2)
      INTEGER*4  VAL_I4_ARR(8192), USER_SUP, USER_REC, AUTO_SUP(GVH__MOBS)
      REAL*8     VAL_R8_ARR(8192), VAL_R8, RWDELVAL(0:GVF_DB__MSLT-1,MAX_ARC_BSL), &
     &                                     RWRATVAL(0:GVF_DB__MSLT-1,MAX_ARC_BSL), &
     &           UTC_OBS(GVH__MOBS), UTC_TAG, UTC_M_TAI, NUM_SAM1(GVH__MSTA), &
     &           NUM_SAM2(GVH__MSTA), SKYFRQCH(GVF_DB__MFRQ), CLO_INTR(GVF_DB__MSLT), &
     &           ATM_INTR(GVF_DB__MSLT), TIL_INTR(GVF_DB__MSLT), FRQ_MIN, FRQ_MAX, &
     &           SAMPLRAT(16)
      REAL*8,    ALLOCATABLE :: CNS_ARR(:,:)
      REAL*4     UV_CHN(2,GVF_DB__MFRQ)
      LOGICAL*1  FL_CLK, LEX, FL_DEBUG, FL_NUSEDCHN, FL_REW
      INTEGER*2  DELUFLAG, PHSUFLAG, UACSUP(1), IND_CAL_STA(GVH__MSTA)
      CHARACTER  ARR_CH16(MAX_ARC_BSL)*16, C_ARR_CH(GVH__MSTA)*8, QUALCODE(2)*1, &
     &           FILVER*3, NERS_CONFIG*128
      ADDRESS__TYPE ADR_DATA, ADR_CONV
      INTEGER*2 INT2_ARG
      INTEGER*4 INT4
      CHARACTER, EXTERNAL :: MJDSEC_TO_DATE*31
      INTEGER*4, EXTERNAL :: ILEN, I_LEN, LTM_DIF, ADD_CLIST
      LOGICAL*4, EXTERNAL :: IS_R8_NAN
      LOGICAL*2, EXTERNAL :: KBIT
      INT4(INT2_ARG) = INT(INT2_ARG,KIND=4)
!
      FL_DEBUG = .FALSE.
!
! --- Get NERS configuration file
!
      CALL GETENVAR ( 'HOME', NERS_CONFIG )
      NERS_CONFIG = TRIM(NERS_CONFIG)//'/.ners_config'
      INQUIRE ( FILE=NERS_CONFIG, EXIST=LEX )
      IF ( .NOT. LEX ) THEN
           NERS_CONFIG = NERS__CONFIG
      END IF
!
! --- Get the list of LCODEs
!
      CALL ERR_PASS ( IUER, IER )
      CALL GVH_GET_LCODE_LIST ( GVH_NS, GVH__MTOC, L_LCD, C_LCD, IER )
      IF ( IER .NE. 0 ) THEN
           CALL ERR_LOG ( 1901, IUER, 'GVH_FROM_NUSOLVE', 'Error in getting '// &
     &         'the list of lcodes from the nuSolve database file' )
           RETURN 
      END IF
!
! --- Check LCODEs and find whether there is CLOK_SITS 
! --- This coode appears only of the experiment has been analyzed by Solve
! --- Check also for NUSEDCHN code. It may or may not preent
!
      FL_CLK      = .FALSE.
      FL_NUSEDCHN = .FALSE.
      FL_REW      = .FALSE.
      DO 410 J1=1,L_LCD 
         IP = LTM_DIF ( 0, GVH__NLC, GVH__LC, C_LCD(J1) )
         IF ( IP < 1 ) THEN
              CALL ERR_LOG ( 1902, IUER, 'GVH_FROM_NUSOLVE', 'Trap of internal '// &
     &            'control: an unknown lcode '//C_LCD(J1)//' was found in the '// &
     &            'nuSolve database file. Please, add this lcode to GVH__LC array' )
              RETURN 
         END IF
!
         IF ( C_LCD(J1) == 'CLK_SITS' ) FL_CLK      = .TRUE.
         IF ( C_LCD(J1) == 'NUSEDCHN' ) FL_NUSEDCHN = .TRUE.
!
         IF ( FL_DEBUG ) THEN
              WRITE ( 6, 110 ) J1, C_LCD(J1), IP, GVH__TBI(IP)
 110          FORMAT ( I3, 1X, A, ' IP= ', I3, ' tbi= ', I3 )
         END IF
 410  CONTINUE 
!
! --- Prepare the data structure fro reading the LCODEs
!
      CALL ERR_PASS ( IUER, IER )
      CALL GVH_PREGET ( GVH_NS, IER )
      IF ( IER .NE. 0 ) THEN
           CALL ERR_LOG ( 1903, IUER, 'GVH_FROM_NUSOLVE', 'Error in an attempt to '// &
     &         'execute GVH_PREGET' )
           RETURN 
      END IF
!
! --- Read Mandatory LCODES from the NuSolve database
!
      DO 420 J2=1,GVH__NMLCODE
!
! ------ Get information about the LCODE
!
         CALL ERR_PASS   ( IUER, IER )
         CALL GVH_INQ_LCODE ( GVH_NS, GVH__MLCODE(J2), DESCR, CLASS, TYP, DIMS, &
     &                        NUM_FIELDS, SEG_IND, LEN_REC, LEN_DATA, &
     &                        ADR_DATA, IER )
         IF ( IER .NE. 0 ) THEN
              CALL ERR_LOG ( 1904, IUER, 'GVH_FROM_NUSOLVE', 'Error in an attempt to '// &
     &            'execute GVH_PREGET' )
              RETURN 
         END IF
!
! ------ Allocate dynamic memory for the manadatory lcode
!
         CALL GET_MEM ( INT8(LEN_DATA), ADR_DATA )
         IF ( ADR_DATA == 0 ) THEN
              CALL CLRCH ( STR ) 
              CALL IINCH ( LEN_DATA, STR )
              CALL ERR_LOG ( 1905, IUER, 'GVH_FROM_NUSOLVE', 'Error in an attempt to '// &
     &            'allocate '//TRIM(STR)//' bytes of dynamic memory' )
              RETURN 
         END IF
!
         CALL ERR_PASS   ( IUER, IER )
         CALL GVH_GLCODE ( GVH_NS, GVH__MLCODE(J2), 0, 0, LEN_DATA, &
     &                     DIMS(1), DIMS(2), %VAL(ADR_DATA), IER )
         IF ( IER .NE. 0 ) THEN
              CALL ERR_LOG ( 1906, IUER, 'GVH_FROM_NUSOLVE', 'Error in an attempt to '// &
     &            'get data of lcode '//GVH__MLCODE(J2)//' from the input database' )
              RETURN 
         END IF
!
! ------ Copy the contents of the mandatory locde where it is needed
!
         IF ( GVH__MLCODE(J2) == 'NOBS_STA' ) THEN
              CALL MEMCPY ( NOBS_STA, %VAL(ADR_DATA), %VAL(LEN_DATA) )
           ELSE IF ( GVH__MLCODE(J2) == 'NUMB_OBS' ) THEN
              CALL MEMCPY ( NUMOBS,   %VAL(ADR_DATA), %VAL(LEN_DATA) )
           ELSE IF ( GVH__MLCODE(J2) == 'NUMB_SCA' ) THEN
              CALL MEMCPY ( NUMSCA,   %VAL(ADR_DATA), %VAL(LEN_DATA) )
           ELSE IF ( GVH__MLCODE(J2) == 'NUMB_STA' ) THEN
              CALL MEMCPY ( NUMSTA,   %VAL(ADR_DATA), %VAL(LEN_DATA) )
           ELSE IF ( GVH__MLCODE(J2) == 'OBS_TAB ' ) THEN
              CALL MEMCPY ( OBS_TAB,  %VAL(ADR_DATA), %VAL(LEN_DATA) )
           ELSE IF ( GVH__MLCODE(J2) == 'SITNAMES' ) THEN
              CALL MEMCPY ( C_STA,    %VAL(ADR_DATA), %VAL(LEN_DATA) )
         END IF
!
! ------ Free dunamic memory
!
         CALL FREE_MEM ( ADR_DATA )
 420  CONTINUE 
!
! --- Create preamble for the 1st section of the output GVH object
!
      GVH%NSEG = 4
      CALL ERR_PASS ( IUER, IER )
      CALL GVH_PPREA ( GVH, 1, 'DEF_TYPE:',  '1 CHARACTER ASCII', IER )
      IF ( IER .NE. 0 ) THEN
           CALL ERR_LOG ( 1907, IUER, 'GVH_FROM_NUSOLVE', 'Error in writing '// &
     &                   'preamlbe of GVH structure' )
           RETURN 
      END IF
!
      CALL GVH_PPREA ( GVH, 1, 'DEF_TYPE:',    '1 CHARACTER ASCII', IER )
      CALL GVH_PPREA ( GVH, 1, 'DEF_TYPE:',    '2 INTEGER*2 IEEE-231', IER )
      CALL GVH_PPREA ( GVH, 1, 'DEF_TYPE:',    '3 INTEGER*4 IEEE-231', IER )
      CALL GVH_PPREA ( GVH, 1, 'DEF_TYPE:',    '4 REAL*4 IEEE 754-1985', IER )
      CALL GVH_PPREA ( GVH, 1, 'DEF_TYPE:',    '5 REAL*8 IEEE 754-1985', IER )
      CALL GVH_PPREA ( GVH, 1, 'DEF_TYPE:',    '6 INTEGER*8 IEEE-231', IER )
      CALL GVH_PPREA ( GVH, 1, 'DEF_CLASS:',   '81 Session', IER )
      CALL GVH_PPREA ( GVH, 1, 'DEF_CLASS:',   '82 Scan', IER )
      CALL GVH_PPREA ( GVH, 1, 'DEF_CLASS:',   '83 Station', IER )
      CALL GVH_PPREA ( GVH, 1, 'DEF_CLASS:',   '84 Baseline', IER )
      CALL GVH_PPREA ( GVH, 1, 'GVH_VERSION:', GVH__LABEL, IER )
!
      CALL ERR_PASS ( IUER, IER )
      CALL GVH_PPREA ( GVH, 1, 'GENERATOR:', GVH__TRANSFORM_LABEL//' AFTER '//GVH_NS%GENERATOR, IER )
      IF ( IER .NE. 0 ) THEN
           CALL ERR_LOG ( 1908, IUER, 'GVH_FROM_NUSOLVE', 'Error in writing '// &
     &                   'preamlbe of GVH structure' )
           RETURN 
      END IF
!
! --- Process lcodes from the input GVH object
!
      NUM_CLRF = 0
      DO 430 J3=1,L_LCD
         IP = LTM_DIF ( 0, GVH__NLC, GVH__LC, C_LCD(J3) )
!
! ------ Bypass an LCODE with "ignore" status
!
         IF ( GVH__TBI(IP) == GVH__IGN ) GOTO 430
!
! ------ Bypass mandatory code, since it has already been uploaded
!
         IF ( LTM_DIF ( 0, GVH__NMLCODE, GVH__MLCODE, C_LCD(J3) ) > 0 ) GOTO 430
!
! ------ Inquire information about the J3-th lcode
!
         CALL ERR_PASS   ( IUER, IER )
         CALL GVH_INQ_LCODE ( GVH_NS, C_LCD(J3), DESCR, CLASS, TYP, DIMS, &
     &                        NUM_FIELDS, SEG_IND, LEN_REC, LEN_DATA, &
     &                        ADR_DATA, IER )
         IF ( IER .NE. 0 ) THEN
              CALL ERR_LOG ( 1909, IUER, 'GVH_FROM_NUSOLVE', 'Error in an attempt to '// &
     &            'execute GVH_PREGET' )
              RETURN 
         END IF
         IF ( C_LCD(J3) == 'MJD_OBS ' ) THEN
!
! ----------- Get MJD_OBS array. It will be needed later
!
              CALL ERR_PASS   ( IUER, IER )
              CALL GVH_GLCODE ( GVH_NS, C_LCD(J3), 1, 1, LEN_DATA, &
     &                          DIMS(1), DIMS(2), MJD_OBS, IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 1910, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                 'an attempt to get data of lcode "MJD_OBS"'// &
     &                 ' from the input database' )
                   RETURN 
              END IF
         END IF
!
         IF ( C_LCD(J3) == 'UTC_OBS ' ) THEN
!
! ----------- Get UTC_OBS array. It will be needed later
!
              CALL ERR_PASS   ( IUER, IER )
              CALL GVH_GLCODE ( GVH_NS, C_LCD(J3), 1, 1, LEN_DATA, &
     &                          DIMS(1), DIMS(2), UTC_OBS, IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 1911, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                 'an attempt to get data of lcode "UTC_OBS"'// &
     &                 ' from the input database' )
                   RETURN 
              END IF
         END IF
!
         IF ( GVH__TBI(IP) == GVH__PRC ) THEN
!
! ----------- Now we process lcodes in GVH__PRC category, i.e. those that have to
! ----------- be processed
!
              IR = LTM_DIF ( 0, GVH__PRC_NUS, GVH__LC_NUS, C_LCD(J3) )
              IF ( C_LCD(J3) == 'ATM_CNST' ) THEN
!
! ---------------- Change units and dimensions
!
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_PSO(1,IR), TYP, CLASS, NUMSTA, GVF_DB__MSLT, &
     &                            'Reciprocal weights of constraints on atm. path delay rate per station, soltype', &
     &                             GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1912, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       GVH__LC_PSO(1,IR) )
                        RETURN
                   END IF
                ELSE IF ( C_LCD(J3) == 'EXP_CODE' ) THEN
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_PSO(1,IR), TYP, CLASS, DIMS(1), DIMS(2), &
     &                             DESCR, GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1913, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       GVH__LC_PSO(1,IR) )
                        RETURN
                   END IF
                ELSE IF ( C_LCD(J3) == 'CLK_CNST'      ) THEN   
!
! ---------------- Change units and dimensions
!
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_PSO(1,IR), TYP, CLASS, NUMSTA, GVF_DB__MSLT, &
     &                            'Reciprocal weights of constraints on clock rate per station, per solution type', &
     &                             GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1914, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       GVH__LC_PSO(1,IR) )
                        RETURN
                   END IF
                ELSE IF ( C_LCD(J3) == 'BASLSTAT' ) THEN
!
! ---------------- Change dimensions
!
                   DIMS(1) = (NUMSTA*(NUMSTA-1))/2
                   DIMS(2) = 1
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_PSO(1,IR), GVH__I4, CLASS, DIMS(1), DIMS(2), &
     &                             DESCR, GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1915, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       GVH__LC_PSO(1,IR) )
                        RETURN
                   END IF
!
! ---------------- Create the second lcode
!
!
                   DIMS(1) = NUMSTA
                   DIMS(2) = 1
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_PSO(2,IR), GVH__I4, CLASS, DIMS(1), DIMS(2), &
     &                             DESCR, GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1916, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       GVH__LC_PSO(2,IR) )
                        RETURN
                   END IF
                ELSE IF ( C_LCD(J3) == 'BLDEPCKS' ) THEN
!
! ---------------- Change dimensions
!
                   DIMS(1) = (NUMSTA*(NUMSTA-1))/2
                   DIMS(2) = 1
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_PSO(1,IR), GVH__I4, CLASS, DIMS(1), DIMS(2), &
     &                             DESCR, GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1917, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       GVH__LC_PSO(1,IR) )
                        RETURN
                   END IF
                ELSE IF ( C_LCD(J3) == 'CLK_SITS' ) THEN
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_PSO(1,IR), TYP, CLASS, DIMS(1), DIMS(2), &
     &                             DESCR, GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1918, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       GVH__LC_PSO(1,IR) )
                        RETURN
                   END IF
                   NUM_CLRF = DIMS(2)
                ELSE IF ( C_LCD(J3) == 'COHERCOR' .OR. &
     &                    C_LCD(J3) == 'EFF.DURA'      ) THEN
!
! ---------------- Change name
!
                   MLEN_COHERCOR = DIMS(1)*DIMS(2)
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_PSO(1,IR), TYP, CLASS, DIMS(1), DIMS(2), &
     &                             DESCR, GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1919, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       GVH__LC_PSO(1,IR) )
                        RETURN
                   END IF
                ELSE IF ( C_LCD(J3) == 'ERROR_BL' ) THEN
!
! ---------------- Change dimensions
!
                   NUM_BAS_RW = DIMS(2)
                   DIMS(2) = (NUMSTA*(NUMSTA-1))/2
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_PSO(1,IR), TYP, CLASS, DIMS(1), DIMS(2), &
     &                             DESCR, GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1920, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       GVH__LC_PSO(1,IR) )
                        RETURN
                   END IF
                ELSE IF ( C_LCD(J3) == 'ERROR_K ' ) THEN
!
! ---------------- Change dimensions
!
                   NUM_BAS_RW = DIMS(2)
                   DIMS(1) = GVF_DB__MSLT
                   DIMS(2) = (NUMSTA*(NUMSTA-1))/2
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_PSO(1,IR), TYP, CLASS, DIMS(1), DIMS(2), &
     &                             DESCR, GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1921, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       GVH__LC_PSO(1,IR) )
                        RETURN
                   END IF
!
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_PSO(2,IR), TYP, CLASS, DIMS(1), DIMS(2), &
     &                             DESCR, GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1922, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       GVH__LC_PSO(1,IR) )
                        RETURN
                   END IF
                   FL_REW = .TRUE.
                ELSE IF ( C_LCD(J3) == 'SOURSTAT' ) THEN
!
! ---------------- Change dimensions
!
                   NUMSOU = DIMS(1)
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_PSO(1,IR), GVH__I4, CLASS, &
     &                             DIMS(1), DIMS(2), DESCR, GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1923, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       GVH__LC_PSO(1,IR) )
                        RETURN
                   END IF
!
! ---------------- Create the second lcode
!
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_PSO(2,IR), GVH__I4, CLASS, &
     &                             2, DIMS(1), DESCR, GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1924, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       GVH__LC_PSO(2,IR) )
                        RETURN
                   END IF
                ELSE IF ( C_LCD(J3) == 'DELUFLAG' ) THEN
!
! ---------------- Change name
!
                   DESCR = 'Bit field of analyst defined recovery status for combination of observables'
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_PSO(1,IR), TYP, CLASS, DIMS(1), DIMS(2), &
     &                             DESCR, GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1925, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       GVH__LC_PSO(1,IR) )
                        RETURN
                   END IF
                ELSE IF ( C_LCD(J3) == 'UACSUP  ' ) THEN
!
! ---------------- Change name
!
                   DESCR = 'Bit field of analyst defined suppression status for combination of observables'
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_PSO(1,IR), GVH__I4, CLASS, DIMS(1), DIMS(2), &
     &                             DESCR, GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1926, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       GVH__LC_PSO(1,IR) )
                        RETURN
                   END IF
                ELSE IF ( C_LCD(J3) == 'PHSUFLAG' ) THEN
!
! ---------------- Ignore it
!
                   CONTINUE 
                ELSE IF ( C_LCD(J3) == 'CAL_LIST' ) THEN
!
! ---------------- Change names and dimensions
!
                   CALL ERR_PASS   ( IUER, IER )
                   CALL GVH_GLCODE ( GVH_NS, C_LCD(J3), 1, 1, LEN_DATA, &
     &                               DIMS(1), DIMS(2), C_ARR_CH, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1927, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                      'an attempt to get data of lcode '//GVH__LC_NUS(IR)// &
     &                      ' from the input database' )
                        RETURN 
                   END IF
                   IND_CAB_CAL = 0
                   IND_UNP_CAL = 0
                   DO 440 J4=1,DIMS(2)
                      IF ( LTM_DIF ( 0, L_LCD, C_LCD, C_ARR_CH(J4) ) > 0 ) THEN
                           I_CAL = ADD_CLIST ( GVH__MTOC, L_CAL, C_CAL, C_ARR_CH(J4), IER )
                           IF ( C_ARR_CH(J4) == 'CABL_DEL' ) THEN
                                CAL_INFO(I_CAL)%CLASS = GVH__STA
                                CAL_INFO(I_CAL)%MODE  = CAL__DEL
                                IND_CAB_CAL = J4
                             ELSE IF ( C_ARR_CH(J4) == 'UNPHASCL' ) THEN
                                CAL_INFO(I_CAL)%CLASS = GVH__STA
                                CAL_INFO(I_CAL)%MODE  = CAL__DEL
                                IND_UNP_CAL = J4
                           END IF
                      END IF
 440               CONTINUE 
!
                   IF ( L_CAL > 0 ) THEN 
!
! --------------------- If stations calibrations are define, create three lcodes 
!
                        CALL ERR_PASS ( IUER, IER )
                        CALL GVH_PTOC ( GVH, 'N_CALIB ', GVH__I2, CLASS, 1, 1, &
     &                                  'Number of available calibrations', GVH__CL1, IER )
                        IF ( IER .NE. 0 ) THEN
                             CALL ERR_LOG ( 1928, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                           'internal control: cannot insert a definition for lcode '// &
     &                           'N_CALIB ' )
                             RETURN
                        END IF
!
                        CALL ERR_PASS ( IUER, IER )
                        CALL GVH_PTOC ( GVH, 'CAL_NAME', GVH__C1, CLASS, 8, L_CAL, &
     &                                  'Names of available calibrations', GVH__CL1, IER )
                        IF ( IER .NE. 0 ) THEN
                             CALL ERR_LOG ( 1929, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                           'internal control: cannot insert a definition for lcode '// &
     &                           'CAL_NAME' )
                             RETURN
                        END IF
!
                        CALL ERR_PASS ( IUER, IER )
                        CALL GVH_PTOC ( GVH, 'CAL_INFO', GVH__I4, CLASS, 4, L_CAL, &
     &                                  'Information about class and type of '// &
     &                                  'available calibrations', GVH__CL1, IER )
                        IF ( IER .NE. 0 ) THEN
                             CALL ERR_LOG ( 1930, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                           'internal control: cannot insert a definition for lcode '// &
     &                           'CAL_INFO' )
                             RETURN
                        END IF
                   END IF
!
! ---------------- Create a new lcode sith calibration status
!
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, 'CAL_STS ', GVH__I4, GVH__SES, NUMSTA, &
     &                  L_CAL, 'Bit field of using available calibrations '//&
     &                 'per station, per calibration', GVH__SL1, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1931, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                      'CAL_STS' )
                        RETURN
                   END IF
                   CAL_STS = 0
                ELSE IF ( C_LCD(J3) == 'CAL_FLGS' ) THEN
!
! ---------------- Change dimensions
!
!
! ---------------- Change name
!
                   CALL ERR_PASS   ( IUER, IER )
                   CALL GVH_GLCODE ( GVH_NS, C_LCD(J3), 1, 1, LEN_DATA, &
     &                               DIMS(1), DIMS(2), IND_CAL_STA, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1932, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                      'an attempt to get data of lcode '//GVH__LC_NUS(IR)// &
     &                      ' from the input database' )
                        RETURN 
                   END IF
                ELSE IF ( C_LCD(J3) == 'NUM_BAND' ) THEN
!
! ---------------- Change name
!
                   CALL ERR_PASS   ( IUER, IER )
                   CALL GVH_GLCODE ( GVH_NS, C_LCD(J3), 1, 1, LEN_DATA, &
     &                               DIMS(1), DIMS(2), NUMBAND, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1933, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                      'an attempt to get data of lcode '//GVH__LC_NUS(IR)// &
     &                      ' from the input database' )
                        RETURN 
                   END IF
!
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_NUS(IR), TYP, CLASS, DIMS(1), DIMS(2), &
     &                             DESCR, GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1934, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       GVH__LC_PSO(1,IR) )
                        RETURN
                   END IF
                   IF ( LTM_DIF ( 0, L_LCD, C_LCD, GVH__LC_PSO(1,IR) ) .LE. 0 ) THEN
!
! --------------------- Create the second lcode if it does not exist
!
                        CALL ERR_PASS ( IUER, IER )
                        CALL GVH_PTOC ( GVH, GVH__LC_PSO(1,IR), TYP, CLASS, DIMS(1), &
     &                                  DIMS(2), 'Number of frequency bands for '// &
     &                                 'which information is availble', &
     &                                  GVH__PRC_TBI(IR), IER )
                        IF ( IER .NE. 0 ) THEN
                             CALL ERR_LOG ( 1935, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                           'internal control: cannot insert a definition for lcode '// &
     &                           GVH__LC_PSO(1,IR) )
                             RETURN
                        END IF
                   END IF
                ELSE IF ( C_LCD(J3) == 'NSAMPLS1' ) THEN
!
! ---------------- Change name
!
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_PSO(1,IR), TYP, CLASS, DIMS(1), DIMS(2), &
     &                             DESCR, GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1936, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                      GVH__LC_PSO(1,IR) )
                        RETURN
                   END IF
                ELSE IF ( C_LCD(J3) == 'NSAMPLS2' ) THEN
!
! ---------------- Change name
!
                   CALL ERR_PASS   ( IUER, IER )
                   CALL GVH_GLCODE ( GVH_NS, C_LCD(J3), 1, 1, LEN_DATA, &
     &                               DIMS(1), DIMS(2), NUM_SAM2, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1937, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                      'an attempt to get data of lcode '//GVH__LC_NUS(IR)// &
     &                      ' from the input database' )
                        RETURN 
                   END IF
!
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_PSO(1,IR), TYP, CLASS, DIMS(1), DIMS(2), &
     &                             DESCR, GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1938, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                      GVH__LC_PSO(1,IR) )
                        RETURN
                   END IF
                 ELSE IF ( C_LCD(J3) == 'ION_CORR' ) THEN
!
! ---------------- Change name
!
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_PSO(1,IR), TYP, CLASS, 1, 1, &
     &                             DESCR, GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1939, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       GVH__LC_PSO(1,IR) )
                        RETURN
                   END IF
!
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_PSO(2,IR), TYP, CLASS, 1, 1, &
     &                             DESCR, GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1940, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       GVH__LC_PSO(2,IR) )
                        RETURN
                   END IF
                 ELSE IF ( C_LCD(J3) == 'IONRMS  ' ) THEN
!
! ---------------- Change name
!
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_PSO(1,IR), TYP, CLASS, 1, 1, &
     &                             DESCR, GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1941, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       GVH__LC_PSO(1,IR) )
                        RETURN
                   END IF
!
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_PSO(2,IR), TYP, CLASS, 1, 1, &
     &                             DESCR, GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1942, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       GVH__LC_PSO(2,IR) )
                        RETURN
                   END IF
                 ELSE IF ( C_LCD(J3) == 'APBYFRQ1' ) THEN
!
! ---------------- Change name and dimensions
!
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_PSO(1,IR), GVH__R4, CLASS, DIMS(2), DIMS(1), &
     &                             'UV data: real and image part per channel at the 1st band (d/l)', &
     &                              GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1943, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       GVH__LC_PSO(1,IR) )
                       RETURN
                   END IF
                 ELSE IF ( C_LCD(J3) == 'APBYFRQ2' ) THEN
!
! ---------------- Change name and dimensions
!
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_PSO(1,IR), GVH__R4, CLASS, DIMS(2), DIMS(1), &
     &                             'UV data: real and image part per channel at the 2nd band (d/l)', &
     &                              GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1944, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       GVH__LC_PSO(1,IR) )
                       RETURN
                   END IF
                 ELSE IF ( C_LCD(J3) == 'UNPHASCL' ) THEN
!
! ---------------- Change name and dimensions
!
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_PSO(1,IR), GVH__R8, GVH__STA, 1, 1, &
     &                             'Contribution to group delay if to remove phase calibration', &
     &                             GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1945, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       GVH__LC_PSO(1,IR) )
                       RETURN
                   END IF
                ELSE IF ( C_LCD(J3) == 'SCANNAME' ) THEN
!
! ---------------- Change name and dimensions
!
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_PSO(1,IR), GVH__C1, GVH__SCA, 16, 1, &
     &                             'Scan name', GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1946, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       GVH__LC_PSO(1,IR) )
                       RETURN
                   END IF
                ELSE IF ( C_LCD(J3) == 'CLK_INTV' ) THEN
!
! ---------------- Change name and dimensions
!
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_PSO(1,IR), TYP, CLASS, GVF_DB__MSLT, 1, &
     &                             'Lenght of time span between spline nodes for '// &
     &                             'clock function per soltyp (sec)', GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1947, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       GVH__LC_PSO(1,IR) )
                       RETURN
                   END IF
                ELSE IF ( C_LCD(J3) == 'ATM_INTV' ) THEN
!
! ---------------- Change name and dimensions
!
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_PSO(1,IR), TYP, CLASS, GVF_DB__MSLT, 1, &
     &                             'Lenght of time span between spline nodes for '// &
     &                             'atm. path delay per soltyp (sec)', GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1948, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       GVH__LC_PSO(1,IR) )
                       RETURN
                   END IF
                ELSE IF ( C_LCD(J3) == 'RFREQ1  ' ) THEN
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, C_LCD(J3), TYP, CLASS, DIMS(1), DIMS(2), &
     &                             DESCR, GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1949, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       C_LCD(J3) )
                       RETURN
                   END IF
                   IP = DIMS(1)
!
                   CALL ERR_PASS   ( IUER, IER )
                   CALL GVH_GLCODE ( GVH_NS, 'NUM_BAND', 1, 1, LEN_DATA, &
     &                               DIMS(1), DIMS(2), NUMBAND, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1950, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                      'an attempt to get data of lcode NUM_BAND '// &
     &                      ' from the input database' )
                        RETURN 
                   END IF
!
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_PSO(1,IR), GVH__I2, CLASS, IP, DIMS(2), &
     &                             'Indexes of channels used in bandwidth synthesis in band 1', &
     &                             GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1951, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       GVH__LC_PSO(1,IR) )
                       RETURN
                   END IF
!
                   IF ( .NOT. FL_NUSEDCHN ) THEN
                        CALL ERR_PASS ( IUER, IER )
                        CALL GVH_PTOC ( GVH, GVH__LC_PSO(2,IR), GVH__I2, CLASS, NUMBAND, 1, &
     &                                  'Number of channels used in bandwidth synthesis per band'// &
     &                                  'atm. path delay per soltyp (sec)', GVH__PRC_TBI(IR), IER )
                        IF ( IER .NE. 0 ) THEN
                             CALL ERR_LOG ( 1952, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                           'internal control: cannot insert a definition for lcode '// &
     &                            GVH__LC_PSO(2,IR) )
                            RETURN
                        END IF
                   END IF
                ELSE IF ( C_LCD(J3) == 'RFREQ2  ' ) THEN
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, C_LCD(J3), TYP, CLASS, DIMS(1), DIMS(2), &
     &                             DESCR, GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1953, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       C_LCD(J3) )
                       RETURN
                   END IF
!
                   CALL ERR_PASS ( IUER, IER )
                   CALL GVH_PTOC ( GVH, GVH__LC_PSO(1,IR), GVH__I2, CLASS, DIMS(1), DIMS(2), &
     &                             'Indexes of channels used in bandwidth synthesis in band 2', &
     &                             GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1954, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       GVH__LC_PSO(1,IR) )
                       RETURN
                   END IF
                ELSE IF ( C_LCD(J3) == 'QUALCODE' ) THEN
                   CALL ERR_PASS ( IUER, IER )
                   DIMS(1) = 1 
                   DIMS(2) = NUMBAND
                   CALL GVH_PTOC ( GVH, C_LCD(J3), TYP, CLASS, DIMS(1), DIMS(2), &
     &                             DESCR, GVH__PRC_TBI(IR), IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1955, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                      'internal control: cannot insert a definition for lcode '// &
     &                       C_LCD(J3) )
                       RETURN
                   END IF
              END IF
            ELSE
!
! ----------- No changes
!
              CALL ERR_PASS ( IUER, IER )
              CALL GVH_PTOC ( GVH, C_LCD(J3), TYP, CLASS, DIMS(1), DIMS(2), &
     &                        DESCR, GVH__TBI(IP), IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 1956, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                 'internal control: cannot insert a definition for lcode '// &
     &                  C_LCD(J3) )
                  RETURN
              END IF
         END IF
 430  CONTINUE 
!
      IF ( .NOT. FL_REW ) THEN
           NUM_BAS_RW = NUMSTA
           DIMS(1) = GVF_DB__MSLT
           DIMS(2) = (NUMSTA*(NUMSTA-1))/2
!
           IR = LTM_DIF ( 0, GVH__PRC_NUS, GVH__LC_NUS, 'ERROR_K ' )
           CALL ERR_PASS ( IUER, IER )
           CALL GVH_PTOC ( GVH, GVH__LC_PSO(1,IR), TYP, CLASS, DIMS(1), DIMS(2), &
     &                     DESCR, GVH__PRC_TBI(IR), IER )
           IF ( IER .NE. 0 ) THEN
                CALL ERR_LOG ( 1921, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &              'internal control: cannot insert a definition for lcode '// &
     &               GVH__LC_PSO(1,IR) )
                RETURN
           END IF
!
           CALL ERR_PASS ( IUER, IER )
           CALL GVH_PTOC ( GVH, GVH__LC_PSO(2,IR), TYP, CLASS, DIMS(1), DIMS(2), &
     &                     DESCR, GVH__PRC_TBI(IR), IER )
           IF ( IER .NE. 0 ) THEN
                CALL ERR_LOG ( 1922, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &              'internal control: cannot insert a definition for lcode '// &
     &               GVH__LC_PSO(1,IR) )
                RETURN
           END IF
      END IF
!
      IF ( LTM_DIF ( 0, L_LCD, C_LCD, 'NSAMPLS1' ) .LE. 0 ) THEN
           CALL ERR_PASS   ( IUER, IER )
           CALL GVH_GLCODE ( GVH_NS, 'NUM_CHBN', 1, 1, 8, DIMS(1), DIMS(2), NUM_CHNS, IER )
           IF ( IER .NE. 0 ) THEN
                CALL ERR_LOG ( 1957, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &              'an attempt to get data of lcode NUM_CHBN '// &
     &              ' from the input database' )
                RETURN 
           END IF
!
           CALL ERR_PASS ( IUER, IER )
           CALL GVH_PTOC ( GVH, 'NUM_SAM1', GVH__R8, GVH__BAS, NUM_CHNS(1), 2, &
     &                     'Number of samples used in bandwidth synth. '// &
     &                     'in band 1 per freq. chan and sideband', GVH__FR2, IER )
           IF ( IER .NE. 0 ) THEN
                CALL ERR_LOG ( 1958, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &              'internal control: cannot insert a definition for lcode '// &
     &              'NUM_SAM1' ) 
                RETURN
           END IF
      END IF
!
      IF ( NUMBAND > 1 .AND. LTM_DIF ( 0, L_LCD, C_LCD, 'NSAMPLS2' ) .LE. 0 ) THEN
           CALL ERR_PASS   ( IUER, IER )
           CALL GVH_GLCODE ( GVH_NS, 'NUM_CHBN', 1, 1, 8, DIMS(1), DIMS(2), NUM_CHNS, IER )
           IF ( IER .NE. 0 ) THEN
                CALL ERR_LOG ( 1959, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &              'an attempt to get data of lcode NUM_CHBN '// &
     &              ' from the input database' )
                RETURN 
           END IF
!
           IF ( NUM_CHNS(2) == 0 ) NUM_CHNS(2) = 1
           CALL ERR_PASS ( IUER, IER )
           CALL GVH_PTOC ( GVH, 'NUM_SAM2', GVH__R8, GVH__BAS, NUM_CHNS(2), 2, &
     &                     'Number of samples used in bandwidth synth. '// &
     &                     'in band 2 per freq. chan and sideband', GVH__FR2, IER )
           IF ( IER .NE. 0 ) THEN
                WRITE ( 6, * ) ' NUM_CHNS= ', NUM_CHNS
                CALL ERR_LOG ( 1960, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &              'internal control: cannot insert a definition for lcode '// &
     &              'NUM_SAM2' ) 
                RETURN
           END IF
      END IF
!
! --- Define new LCODEs
!
      DO 450 J5=1,GVH__NEW_NUS
         IF ( GVH__LC_NEW(J5) == 'NUM_CLRF' .AND. .NOT. FL_CLK ) GOTO 450
!
         DIMS(1) = GVH__DM_NEW(1,J5)
         DIMS(2) = GVH__DM_NEW(2,J5)
!
         IF ( DIMS(1) == -GVH__SCA ) DIMS(1) = NUMSCA
         IF ( DIMS(1) == -GVH__STA ) DIMS(1) = NUMSTA
         IF ( DIMS(1) == -GVH__BAS ) DIMS(1) = NUMOBS
!
         IF ( LTM_DIF ( 0, L_LCD, C_LCD, GVH__LC_NEW(J5) ) .LE. 0 ) THEN
!
! ----------- Define a new lcods unless it exists
!
              CALL ERR_PASS ( IUER, IER )
              CALL GVH_PTOC ( GVH, GVH__LC_NEW(J5), GVH__TY_NEW(J5), GVH__CL_NEW(J5), &
     &                        DIMS(1), DIMS(2), GVH__DS_NEW(J5), GVH__SG_NEW(J5), IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 1961, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                 'internal control: cannot insert a definition for lcode '// &
     &                  GVH__LC_NEW(J5) )
                   RETURN
              END IF
         END IF
 450  CONTINUE
!
! --- Prepare GVH object for putting new lcodes
!
      CALL ERR_PASS   ( IUER, IER )
      CALL GVH_PREPUT ( GVH, NUMOBS, NUMSCA, NUMSTA, NOBS_STA, C_STA, &
     &                  OBS_TAB, IER )
      IF ( IER .NE. 0 ) THEN
           CALL ERR_LOG ( 1962, IUER, 'GVH_FROM_NUSOLVE', 'Error in an attempt to '// &
     &         'load mandatory lcodes to the output database' )
           RETURN 
      END IF
!
! --- Cycle over the old lcodes that we need to put in the output databse
!
      FRQ_MIN =  1.D12
      FRQ_MAX = -1.D12
!
      CALL ERR_PASS   ( IUER, IER )
      CALL GVH_GLCODE ( GVH_NS, 'SRCNAMES', 0, 0, GVH__MSOU*LEN(C_SOU(1)), &
     &                  DIMS(1), DIMS(2), C_SOU, IER )
      IF ( IER .NE. 0 ) THEN
           CALL ERR_LOG ( 1963, IUER, 'GVH_FROM_NUSOLVE', 'Error in an attempt to '// &
     &         'get data of lcode "SRCNAMES" from the input database' )
           RETURN 
      END IF
      FL_REW = .FALSE.
      DO 460 J6=1,L_LCD
         IP = LTM_DIF ( 0, GVH__NLC, GVH__LC, C_LCD(J6) )
!
! ------ Deal with LCODE with "process" status
!
         IF ( GVH__TBI(IP) == GVH__PRC ) THEN
!
! ----------- Get the indes of the lcode in PRC lcodes 
!
              IR = LTM_DIF ( 0, GVH__PRC_NUS, GVH__LC_NUS, C_LCD(J6) )
              IF ( C_LCD(J6) == 'ATM_CNST' .OR. &
     &             C_LCD(J6) == 'CLK_CNST' ) THEN
!
! ---------------- Get the array with atmosphere or clock constraint
!
                   CALL ERR_PASS   ( IUER, IER )
                   CALL GVH_GLCODE ( GVH_NS, GVH__LC_NUS(IR), 0, 0, LEN_DATA, &
     &                               DIMS(1), DIMS(2), VAL_R8_ARR, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1964, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                      'an attempt to get data of lcode '//GVH__LC_NUS(IR)// &
     &                      ' from the input database' )
                        RETURN 
                   END IF
!
! ---------------- Allocate dynamic memory for the output lcode and initialize it
!
                   ALLOCATE ( CNS_ARR(NUMSTA,0:GVF_DB__MSLT-1) )
                   CNS_ARR = 0.0D0
!
                   DO 470 J7=1,NUMSTA
                      DO 480 J8=0,GVF_DB__MSLT-1
                         IF ( J8 == GRPRAT__DTP .OR. &
     &                        J8 == SNBRAT__DTP .OR. &
     &                        J8 == GRPONL__DTP .OR. &
     &                        J8 == SNBONL__DTP .OR. &
     &                        J8 == RATONL__DTP .OR. &
     &                        J8 ==  G_GXS__DTP .OR. &
     &                        J8 ==     GX__DTP .OR. &
     &                        J8 ==     GS__DTP .OR. &
     &                        J8 ==  SNG_X__DTP .OR. &
     &                        J8 ==  SNG_S__DTP      ) THEN
!
! --------------------------- Define the valie of the constraint for the default 
! --------------------------- solution types and update units
!
                              IF ( C_LCD(J6) == 'ATM_CNST' ) THEN
                                   CNS_ARR(J7,J8) = VAL_R8_ARR(J7)*1.D-12/3600.0D0
                                 ELSE IF ( C_LCD(J6) == 'CLK_CNST' ) THEN
                                   CNS_ARR(J7,J8) = VAL_R8_ARR(J7)*1.D-14
                              END IF
                         END IF
 480                  CONTINUE 
 470               CONTINUE 
!
! ---------------- Put the output lcode in the output database
!
                   CALL ERR_PASS   ( IUER, IER )
                   CALL GVH_PLCODE ( GVH, GVH__LC_PSO(1,IR), 0, 0, CNS_ARR, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1965, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                      'an attempt to put data of lcode '//GVH__LC_PSO(1,IR)// &
     &                      ' from the output database' )
                        RETURN 
                   END IF
!
! ---------------- Deallocate memory for the temporary array
!
                   DEALLOCATE ( CNS_ARR )
                 ELSE IF ( C_LCD(J6) == 'BASLSTAT' ) THEN
                   CALL ERR_PASS   ( IUER, IER )
                   CALL GVH_GLCODE ( GVH_NS, GVH__LC_NUS(IR), 1, 1, LEN_DATA, &
     &                               DIMS(1), DIMS(2), BUSE_I2, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1966, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                      'an attempt to get data of lcode '//GVH__LC_NUS(IR)// &
     &                      ' from the input database' )
                        RETURN 
                   END IF
                   BAS_USE = 0
                   STA_USE = 0
                   DO 490 J9=1,NUMSTA-1
                      DO 4100 J10=J9+1,NUMSTA
                         IP  = MAX(J9,J10)-1 + &
     &                         ((MIN(J9,J10)-1)*(2*NUMSTA-2-MIN(J9,J10)))/2
                         KP = (J9-1)*NUMSTA + J10
!
                         DO 4110 J11=0,GVF_DB__MSLT-1
                            IF ( J11 == GRPRAT__DTP .OR. &
     &                           J11 == SNBRAT__DTP .OR. &
     &                           J11 == GRPONL__DTP .OR. &
     &                           J11 == SNBONL__DTP .OR. &
     &                           J11 == RATONL__DTP .OR. &
     &                           J11 ==  G_GXS__DTP .OR. &
     &                           J11 ==     GX__DTP .OR. &
     &                           J11 ==     GS__DTP .OR. &
     &                           J11 ==  SNG_X__DTP .OR. &
     &                           J11 ==  SNG_S__DTP      ) THEN
!
! ------------------------------ Default: to use the baseline
!
                                 BAS_USE(IP)  = IBSET ( BAS_USE(IP), J11 )
                                 IF ( KBIT ( BUSE_I2(KP), INT2(2) ) ) THEN
                                      BAS_USE(IP)  = IBSET ( BAS_USE(IP),  J11 )
                                      STA_USE(J9)  = IBSET ( STA_USE(J9),  J11 )
                                      STA_USE(J10) = IBSET ( STA_USE(J10), J11 )
                                 END IF
                                 IF ( KBIT ( BUSE_I2(KP), INT2(4) ) ) THEN
                                      BSCL_EST(IP) = IBSET ( BSCL_EST(IP), J11 )
                                 END IF
                            END IF
 4110                    CONTINUE 
 4100                 CONTINUE 
 490               CONTINUE
!                       
                   CALL ERR_PASS   ( IUER, IER )
                   CALL GVH_PLCODE ( GVH, GVH__LC_PSO(1,IR), 0, 0, BAS_USE, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1967, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                       'an attempt to put data of lcode '//GVH__LC_PSO(1,IR)// &
     &                       ' from the output database' )
                        RETURN 
                   END IF
!                       
                   CALL ERR_PASS   ( IUER, IER )
                   CALL GVH_PLCODE ( GVH, GVH__LC_PSO(2,IR), 0, 0, STA_USE, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1968, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                       'an attempt to put data of lcode '//GVH__LC_PSO(2,IR)// &
     &                       ' from the output database' )
                        RETURN 
                   END IF
                 ELSE IF ( C_LCD(J6) == 'BLDEPCKS' ) THEN
                   CALL ERR_PASS   ( IUER, IER )
                   CALL GVH_GLCODE ( GVH_NS, GVH__LC_NUS(IR), 1, 1, LEN_DATA, &
     &                               DIMS(1), DIMS(2), ARR_CH16, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1969, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                      'an attempt to get data of lcode '//GVH__LC_NUS(IR)// &
     &                      ' from the input database' )
                        RETURN 
                   END IF
                   CALL NOUT_I4 ( (NUMSTA*(NUMSTA-1))/2, BSCL_EST )
                   DO 4120 J12=1,DIMS(2)
                      IP1 = LTM_DIF ( 0, NUMSTA, C_STA, ARR_CH16(J12)(1:8)  )
                      IP2 = LTM_DIF ( 0, NUMSTA, C_STA, ARR_CH16(J12)(9:16) )
                      IF ( IP1 < 1 ) THEN
                           CALL ERR_LOG ( 1970, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                         'internal control: station '//ARR_CH16(J12)(1:8)// &
     &                         'specified in BLDEPCKS did not observe' )
                          RETURN 
                      END IF
                      IF ( IP2 < 1 ) THEN
                           CALL ERR_LOG ( 1971, IUER, 'GVH_FROM_NUSOLVE', 'Trap of '// &
     &                         'internal control: station '//ARR_CH16(J12)(1:8)// &
     &                         'specified in BLDEPCKS did not observe' )
                          RETURN 
                      END IF
                      IP  =  MAX(IP1,IP2)-1 + &
     &                     ((MIN(IP1,IP2)-1)*(2*NUMSTA-2-MIN(IP1,IP2)))/2
                      DO 4130 J13=0,GVF_DB__MSLT-1
                         IF ( J13 == GRPRAT__DTP .OR. &
     &                        J13 == SNBRAT__DTP .OR. &
     &                        J13 == GRPONL__DTP .OR. &
     &                        J13 == SNBONL__DTP .OR. &
     &                        J13 == RATONL__DTP .OR. &
     &                        J13 ==  G_GXS__DTP .OR. &
     &                        J13 ==     GX__DTP .OR. &
     &                        J13 ==     GS__DTP .OR. &
     &                        J13 ==  SNG_X__DTP .OR. &
     &                        J13 ==  SNG_S__DTP       ) THEN
                              BSCL_EST(IP) = IBSET ( BSCL_EST(IP), J13 )
                         END IF
 4130                 CONTINUE 
 4120              CONTINUE 
!                       
                   CALL ERR_PASS   ( IUER, IER )
                   CALL GVH_PLCODE ( GVH, GVH__LC_PSO(1,IR), 0, 0, BSCL_EST, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1972, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                       'an attempt to put data of lcode '//GVH__LC_PSO(1,IR)// &
     &                       ' from the output database' )
                        RETURN 
                   END IF
                 ELSE IF ( C_LCD(J6) == 'CLK_SITS' ) THEN
                   CALL ERR_PASS   ( IUER, IER )
                   CALL GVH_GLCODE ( GVH_NS, GVH__LC_NUS(IR), 1, 1, LEN_DATA, &
     &                               DIMS(1), DIMS(2), C_ARR_CH, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1973, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                      'an attempt to get data of lcode '//GVH__LC_NUS(IR)// &
     &                      ' from the input database' )
                        RETURN 
                   END IF
!                       
                   CALL ERR_PASS   ( IUER, IER )
                   CALL GVH_PLCODE ( GVH, GVH__LC_PSO(1,IR), 0, 0, C_ARR_CH, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1974, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                       'an attempt to put data of lcode '//GVH__LC_PSO(1,IR)// &
     &                       ' from the output database' )
                        RETURN 
                   END IF
                 ELSE IF ( C_LCD(J6) == 'COHERCOR' .OR. &
     &                     C_LCD(J6) == 'EFF.DURA'      ) THEN
                   DO 4140 J14=1,NUMOBS
                       CALL ERR_PASS   ( IUER, IER )
                       CALL GVH_GLCODE ( GVH_NS, GVH__LC_NUS(IR), J14, 0, 8*MLEN_COHERCOR, &
     &                                   DIMS(1), DIMS(2), VAL_R8_ARR, IER )
                       IF ( IER .NE. 0 ) THEN
                            CALL ERR_LOG ( 1975, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                          'an attempt to get data of lcode '//GVH__LC_NUS(IR)// &
     &                          ' from the input database' )
                            RETURN 
                       END IF
!                       
                       CALL ERR_PASS   ( IUER, IER )
                       CALL GVH_PLCODE ( GVH, GVH__LC_PSO(1,IR), J14, 0, VAL_R8_ARR, IER )
                       IF ( IER .NE. 0 ) THEN
                            CALL ERR_LOG ( 1976, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                          'an attempt to put data of lcode '//GVH__LC_PSO(1,IR)// &
     &                          ' from the output database' )
                            RETURN 
                       END IF
 4140              CONTINUE 
                 ELSE IF ( C_LCD(J6) == 'ERROR_BL' ) THEN
                   KP = 0
                   DO 4150 J15=1,NUMSTA-1
                      DO 4160 J16=J15+1,NUMSTA
                         KP = KP + 1
                         ARR_CH16(KP) = C_STA(J15)//C_STA(J16)
 4160                 CONTINUE 
 4150              CONTINUE 
!
                   CALL ERR_PASS   ( IUER, IER )
                   CALL GVH_PLCODE ( GVH, GVH__LC_PSO(1,IR), 1, 1, ARR_CH16, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1977, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                      'an attempt to put data of lcode '//GVH__LC_PSO(1,IR)// &
     &                      ' from the output database' )
                        RETURN 
                   END IF
                 ELSE IF ( C_LCD(J6) == 'ERROR_K ' ) THEN
                   CALL ERR_PASS   ( IUER, IER )
                   CALL GVH_GLCODE ( GVH_NS, 'ERROR_BL', 1, 1, MAX_ARC_BSL*16, &
     &                               DIMS(1), DIMS(2), ARR_CH16, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1978, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                      'an attempt to get data of lcode '//GVH__LC_NUS(IR)// &
     &                      ' from the input database' )
                        RETURN 
                   END IF
!
                   CALL ERR_PASS   ( IUER, IER )
                   CALL GVH_GLCODE ( GVH_NS, GVH__LC_NUS(IR), 1, 1, MAX_ARC_BSL*8, &
     &                               DIMS(1), DIMS(2), VAL_R8_ARR, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1979, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                      'an attempt to get data of lcode '//GVH__LC_NUS(IR)// &
     &                      ' from the input database' )
                        RETURN 
                   END IF
!
                   NUM_BAS_RW = DIMS(2)
                   RWDELVAL = 0.0D0
                   RWRATVAL = 0.0D0
                   DO 4170 J17=1,NUM_BAS_RW
                      IP1 = LTM_DIF ( 0, NUMSTA, C_STA, ARR_CH16(J17)(1:8)  )
                      IP2 = LTM_DIF ( 0, NUMSTA, C_STA, ARR_CH16(J17)(9:16) )
                      IP  = MAX(IP1,IP2)-1 + &
     &                      ((MIN(IP1,IP2)-1)*(2*NUMSTA-2-MIN(IP1,IP2)))/2
                      DO 4180 J18=0,GVF_DB__MSLT-1
                         IF ( J18 == GRPRAT__DTP .OR. &
     &                        J18 == SNBRAT__DTP .OR. &
     &                        J18 == GRPONL__DTP .OR. &
     &                        J18 == SNBONL__DTP .OR. &
     &                        J18 == RATONL__DTP .OR. &
     &                        J18 ==  G_GXS__DTP .OR. &
     &                        J18 ==     GX__DTP .OR. &
     &                        J18 ==     GS__DTP .OR. &
     &                        J18 ==  SNG_X__DTP .OR. &
     &                        J18 ==  SNG_S__DTP       ) THEN
                              IF ( IP > 0 ) THEN
                                   RWDELVAL(J18,IP) = VAL_R8_ARR(1+2*(J17-1))
                                   RWRATVAL(J18,IP) = VAL_R8_ARR(2+2*(J17-1))
                              END IF
                         END IF
 4180                 CONTINUE 
 4170              CONTINUE 
!
                   CALL ERR_PASS   ( IUER, IER )
                   CALL GVH_PLCODE ( GVH, GVH__LC_PSO(1,IR), 1, 1, RWDELVAL, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1980, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                      'an attempt to put data of lcode '//GVH__LC_PSO(1,IR)// &
     &                      ' from the output database' )
                        RETURN 
                   END IF
!
                   CALL ERR_PASS   ( IUER, IER )
                   CALL GVH_PLCODE ( GVH, GVH__LC_PSO(2,IR), 1, 1, RWRATVAL, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1981, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                      'an attempt to put data of lcode '//GVH__LC_PSO(2,IR)// &
     &                      ' from the output database' )
                        RETURN 
                   END IF
                   FL_REW = .TRUE.
                 ELSE IF ( C_LCD(J6) == 'SOURSTAT' ) THEN
                   CALL GVH_GLCODE ( GVH_NS, GVH__LC_NUS(IR), 1, 1, LEN_DATA, &
     &                               DIMS(1), DIMS(2), VAL_I2_ARR, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1982, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                      'an attempt to get data of lcode '//GVH__LC_NUS(IR)// &
     &                      ' from the input database' )
                        RETURN 
                   END IF
                   SOU_USE  = 0
                   SOCO_EST = 0
                   DO 4190 J19=1,NUMSOU
                      DO 4200 J20=FIRST__DTP,LAST__DTP
                         IF ( KBIT ( VAL_I2_ARR(J19), INT2(1) ) ) THEN
                              SOU_USE(J19)  = IBSET ( SOU_USE(J19), J20 )
                         END IF
                         IF ( KBIT ( VAL_I2_ARR(J19), INT2(3) ) ) THEN
                              SOCO_EST(1,J19) = IBSET ( SOCO_EST(1,J19), J20 )
                         END IF
                         IF ( KBIT ( VAL_I2_ARR(J19), INT2(4) ) ) THEN
                              SOCO_EST(2,J19) = IBSET ( SOCO_EST(2,J19), J20 )
                         END IF
 4200                 CONTINUE 
 4190              CONTINUE 
!
                   CALL ERR_PASS   ( IUER, IER )
                   CALL GVH_PLCODE ( GVH, GVH__LC_PSO(1,IR), 1, 1, SOU_USE, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1983, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                      'an attempt to put data of lcode '//GVH__LC_PSO(1,IR)// &
     &                      ' from the output database' )
                        RETURN 
                   END IF
!
                   CALL ERR_PASS   ( IUER, IER )
                   CALL GVH_PLCODE ( GVH, GVH__LC_PSO(2,IR), 1, 1, SOCO_EST, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 1984, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                      'an attempt to put data of lcode '//GVH__LC_PSO(2,IR)// &
     &                      ' from the output database' )
                        RETURN 
                   END IF
                 ELSE IF ( C_LCD(J6) == 'DELUFLAG' ) THEN
                   DO 4210 J21=1,NUMOBS
                      CALL ERR_PASS   ( IUER, IER )
                      CALL GVH_GLCODE ( GVH_NS, 'DELUFLAG', J21, 1, LEN_DATA, &
     &                                  DIMS(1), DIMS(2), DELUFLAG, IER )
                      IF ( IER .NE. 0 ) THEN
                           CALL ERR_LOG ( 1985, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                         'an attempt to get data of lcode DELUFLAG'// &
     &                         ' from the input database' )
                           RETURN 
                      END IF
!
                      CALL ERR_PASS   ( IUER, IER )
                      CALL GVH_GLCODE ( GVH_NS, 'PHSUFLAG', J21, 1, LEN_DATA, &
     &                                  DIMS(1), DIMS(2), PHSUFLAG, IER )
                      IF ( IER .NE. 0 ) THEN
                           CALL ERR_LOG ( 1986, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                         'an attempt to get data of lcode PHSUFLAG'// &
     &                         ' from the input database' )
                           RETURN 
                      END IF
!
                      CALL ERR_PASS   ( IUER, IER )
                      CALL GVH_GLCODE ( GVH_NS, 'UACSUP  ', J21, 1, LEN_DATA, &
     &                                  DIMS(1), DIMS(2), UACSUP, IER )
                      IF ( IER .NE. 0 ) THEN
                           CALL ERR_LOG ( 1987, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                         'an attempt to get data of lcode UACSUP'// &
     &                         ' from the input database' )
                           RETURN 
                      END IF
!
                      CALL ERR_PASS   ( IUER, IER )
                      CALL GVH_GLCODE ( GVH_NS, 'QUALCODE', J21, 1, LEN_DATA, &
     &                                  DIMS(1), DIMS(2), STR, IER )
                      IF ( IER .NE. 0 ) THEN
                           CALL ERR_LOG ( 1988, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                         'an attempt to get data of lcode DELUFLAG'// &
     &                         ' from the input database' )
                           RETURN 
                      END IF
!
                      IF ( DIMS(1) == 1 ) THEN
                           QUALCODE(1) = STR(1:1)
                           QUALCODE(2) = STR(2:2)
                         ELSE
                           QUALCODE(1) = STR(1:1)
                           QUALCODE(2) = STR(3:3)
                      END IF
                      USER_SUP = 0
                      IF ( QUALCODE(1)(1:1) == ' ' .OR. QUALCODE(1)(1:1) == '-' ) THEN
                           QUALCODE(1)(1:1) = '0'
                      END IF
                      IF ( QUALCODE(2)(1:1) == ' ' .OR. QUALCODE(2)(1:1) == '-' ) THEN
                           QUALCODE(2)(1:1) = '0'
                      END IF
                      IF ( DELUFLAG > 0 .OR. QUALCODE(1)(1:1) == '0' .OR.    &     
     &                     ( ICHAR(QUALCODE(1)) .GE. ICHAR('A') .AND.   &     
     &                       ICHAR(QUALCODE(1)) .LE. ICHAR('F')       ) ) THEN
                           USER_SUP = IBSET ( USER_SUP, GRPONL__DTP )
                           USER_SUP = IBSET ( USER_SUP, GRPRAT__DTP )
                           USER_SUP = IBSET ( USER_SUP, SNBRAT__DTP )
                           USER_SUP = IBSET ( USER_SUP, GRPONL__DTP )
                           USER_SUP = IBSET ( USER_SUP, SNBONL__DTP )
                           USER_SUP = IBSET ( USER_SUP, RATONL__DTP )
                           USER_SUP = IBSET ( USER_SUP, G_GXS__DTP  )
                           USER_SUP = IBSET ( USER_SUP, GX__DTP     )
                           USER_SUP = IBSET ( USER_SUP, GS__DTP     )
                           USER_SUP = IBSET ( USER_SUP, SNG_X__DTP  )
                           USER_SUP = IBSET ( USER_SUP, SNG_S__DTP  )
                      END IF
                      IF (   QUALCODE(2)(1:1) == '0' .OR.             &
     &                     ( ICHAR(QUALCODE(2)) .GE. ICHAR('A') .AND. &
     &                       ICHAR(QUALCODE(2)) .LE. ICHAR('F')       ) ) THEN
                           USER_SUP = IBSET ( USER_SUP, G_GXS__DTP  )
                           USER_SUP = IBSET ( USER_SUP, GS__DTP     )
                           USER_SUP = IBSET ( USER_SUP, SNG_S__DTP  )
                      END IF
                      USER_REC = 0
!
                      CALL ERR_PASS   ( IUER, IER )
                      CALL GVH_PLCODE ( GVH, 'USER_SUP', J21, 1, USER_SUP, IER )
                      IF ( IER .NE. 0 ) THEN
                           CALL ERR_LOG ( 1989, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                         'putting "USER_SUP" lcode' )
                           RETURN
                      END IF
!
                      USER_REC = 0
                      CALL ERR_PASS   ( IUER, IER )
                      CALL GVH_PLCODE ( GVH, 'USER_REC', J21, 1, USER_REC, IER )
                      IF ( IER .NE. 0 ) THEN
                           CALL ERR_LOG ( 1990, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                         'putting "USER_REC" lcode' )
                           RETURN
                      END IF
!
                      CALL ERR_PASS   ( IUER, IER )
                      CALL GVH_PLCODE ( GVH, 'QUALCODE', J21, 1, QUALCODE, IER )
                      IF ( IER .NE. 0 ) THEN
                           CALL ERR_LOG ( 1991, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                         'putting "QUALCODE" lcode' )
                           RETURN
                      END IF
 4210              CONTINUE 
                 ELSE IF ( C_LCD(J6) == 'PHSUFLAG' ) THEN
                    CONTINUE 
                 ELSE IF ( C_LCD(J6) == 'UACSUP  ' ) THEN
                    CONTINUE 
                 ELSE IF ( C_LCD(J6) == 'CAL_LIST' ) THEN
                    IF ( L_CAL > 0 ) THEN 
                         CALL ERR_PASS   ( IUER, IER )
                         CALL GVH_PLCODE ( GVH, 'N_CALIB ', 1, 0, INT2(L_CAL), IER )
                         IF ( IER .NE. 0 ) THEN
                              CALL ERR_LOG ( 1992, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                            'putting "N_CALIB" lcode' )
                              RETURN
                         END IF
!
                         CALL ERR_PASS   ( IUER, IER )
                         CALL GVH_PLCODE ( GVH, 'CAL_NAME', 1, 0, C_CAL, IER )
                         IF ( IER .NE. 0 ) THEN
                              CALL ERR_LOG ( 1993, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                            'putting "CAL_NAME" lcode' )
                              RETURN
                         END IF
!
                         CALL ERR_PASS   ( IUER, IER )
                         CALL GVH_PLCODE ( GVH, 'CAL_INFO', 1, 0, CAL_INFO, IER )
                         IF ( IER .NE. 0 ) THEN
                              CALL ERR_LOG ( 1994, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                            'putting "CAL_INFO" lcode' )
                              RETURN
                         END IF
                    END IF 
                 ELSE IF ( C_LCD(J6) == 'CAL_FLGS' ) THEN
                    DO 4220 J22=1,NUMSTA
                       IF ( IND_CAB_CAL > 0 ) THEN
                            IF ( KBIT ( IND_CAL_STA(J22), INT2(IND_CAB_CAL) ) ) THEN
!
! ------------------------------ Set status bit in the output bit field
!
                                 KP = J22 + (IND_CAB_CAL-1)*NUMSTA
                                 DO 4230 J23=FIRST__DTP,LAST__DTP
                                    CAL_STS(KP) = IBSET ( CAL_STS(KP), J23 )
 4230                            CONTINUE
                                 CAL_STS(KP) = IBSET ( CAL_STS(KP), GVF_DB__MSLT-1 )
                            END IF
                       END IF
                       IF ( IND_UNP_CAL > 0 ) THEN
                            IF ( KBIT ( IND_CAL_STA(J22), INT2(IND_UNP_CAL) ) ) THEN
!
! ------------------------------ Set status bit in the output bit field
!
                                 KP = J22 + (IND_UNP_CAL-1)*NUMSTA
                                 DO 4240 J24=FIRST__DTP,LAST__DTP
                                    CAL_STS(KP) = IBSET ( CAL_STS(KP), J24 )
 4240                            CONTINUE
                                 CAL_STS(KP) = IBSET ( CAL_STS(KP), GVF_DB__MSLT-1 )
                            END IF
                       END IF
 4220               CONTINUE 
!
                    CALL ERR_PASS   ( IUER, IER )
                    CALL GVH_PLCODE ( GVH, 'CAL_STS ', 1, 0, CAL_STS, IER )
                    IF ( IER .NE. 0 ) THEN
                         CALL ERR_LOG ( 1995, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                       'putting "CAL_STS" lcode' )
                         RETURN
                    END IF
                 ELSE IF ( C_LCD(J6) == 'NUM_BAND' ) THEN
                    CALL ERR_PASS   ( IUER, IER )
                    CALL GVH_PLCODE ( GVH, 'NUM_BAND', 1, 0, NUMBAND, IER )
                    IF ( IER .NE. 0 ) THEN
                         CALL ERR_LOG ( 1996, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                       'putting "NUMBAND" lcode' )
                         RETURN
                    END IF
!
                    CALL ERR_PASS   ( IUER, IER )
                    CALL GVH_PLCODE ( GVH, 'N_AVBAND', 1, 0, NUMBAND, IER )
                    IF ( IER .NE. 0 ) THEN
                         CALL ERR_LOG ( 1997, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                       'putting "N_AVBAND" lcode' )
                         RETURN
                    END IF
                 ELSE IF ( C_LCD(J6) == 'NSAMPLS1' ) THEN
                    DO 4250 J25=1,NUMOBS
                       CALL ERR_PASS   ( IUER, IER )
                       CALL GVH_GLCODE ( GVH_NS, C_LCD(J6), J25, 1, LEN_DATA, &
     &                                   DIMS(1), DIMS(2), VAL_R8_ARR, IER )
                       IF ( IER .NE. 0 ) THEN
                            CALL ERR_LOG ( 1998, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                           'an attempt to get data of lcode '//C_LCD(J6)// &
     &                           ' from the input database' )
                            RETURN 
                       END IF
!
                       CALL ERR_PASS   ( IUER, IER )
                       CALL GVH_PLCODE ( GVH, GVH__LC_PSO(1,IR), J25, 1, VAL_R8_ARR, IER )
                       IF ( IER .NE. 0 ) THEN
                            CALL ERR_LOG ( 1999, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                          'putting "NUM_SAM1" lcode' )
                            RETURN
                       END IF
 4250               CONTINUE 
                 ELSE IF ( C_LCD(J6) == 'NSAMPLS2' ) THEN
                    DO 4260 J26=1,NUMOBS
                       CALL ERR_PASS   ( IUER, IER )
                       CALL GVH_GLCODE ( GVH_NS, C_LCD(J6), J26, 1, LEN_DATA, &
     &                                   DIMS(1), DIMS(2), VAL_R8_ARR, IER )
                       IF ( IER .NE. 0 ) THEN
                            CALL ERR_LOG ( 2000, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                           'an attempt to get data of lcode '//C_LCD(J6)// &
     &                           ' from the input database' )
                            RETURN 
                       END IF
!
                       CALL ERR_PASS   ( IUER, IER )
                       CALL GVH_PLCODE ( GVH, GVH__LC_PSO(1,IR), J26, 1, VAL_R8_ARR, IER )
                       IF ( IER .NE. 0 ) THEN
                            CALL ERR_LOG ( 2001, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                          'putting "NUM_SAM2" lcode' )
                            RETURN
                       END IF
 4260               CONTINUE 
                 ELSE IF ( C_LCD(J6) == 'ION_CORR' ) THEN
                    DO 4270 J27=1,NUMOBS
                       CALL ERR_PASS   ( IUER, IER )
                       CALL GVH_GLCODE ( GVH_NS, C_LCD(J6), J27, 1, LEN_DATA, &
     &                                   DIMS(1), DIMS(2), VAL_R8_ARR, IER )
                       IF ( IER .NE. 0 ) THEN
                            CALL ERR_LOG ( 2002, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                           'an attempt to get data of lcode '//C_LCD(J6)// &
     &                           ' from the input database' )
                            RETURN 
                       END IF
!
                       CALL ERR_PASS   ( IUER, IER )
                       CALL GVH_PLCODE ( GVH, GVH__LC_PSO(1,IR), J27, 1, VAL_R8_ARR(1), IER )
                       IF ( IER .NE. 0 ) THEN
                            CALL ERR_LOG ( 2003, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                          'putting '//GVH__LC_PSO(1,IR)//' lcode' )
                            RETURN
                       END IF
!
                       CALL ERR_PASS   ( IUER, IER )
                       CALL GVH_PLCODE ( GVH, GVH__LC_PSO(2,IR), J27, 1, VAL_R8_ARR(2), IER )
                       IF ( IER .NE. 0 ) THEN
                            CALL ERR_LOG ( 2004, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                          'putting '//GVH__LC_PSO(2,IR)//' lcode' )
                            RETURN
                       END IF
 4270              CONTINUE 
                 ELSE IF ( C_LCD(J6) == 'IONRMS  ' ) THEN
                    DO 4280 J28=1,NUMOBS
                       CALL ERR_PASS   ( IUER, IER )
                       CALL GVH_GLCODE ( GVH_NS, C_LCD(J6), J28, 1, LEN_DATA, &
     &                                   DIMS(1), DIMS(2), VAL_R8_ARR, IER )
                       IF ( IER .NE. 0 ) THEN
                            CALL ERR_LOG ( 2005, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                           'an attempt to get data of lcode '//C_LCD(J6)// &
     &                           ' from the input database' )
                            RETURN 
                       END IF
!
                       CALL ERR_PASS   ( IUER, IER )
                       CALL GVH_PLCODE ( GVH, GVH__LC_PSO(1,IR), J28, 1, VAL_R8_ARR(1), IER )
                       IF ( IER .NE. 0 ) THEN
                            CALL ERR_LOG ( 2006, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                          'putting '//GVH__LC_PSO(1,IR)//' lcode' )
                            RETURN
                       END IF
!
                       CALL ERR_PASS   ( IUER, IER )
                       CALL GVH_PLCODE ( GVH, GVH__LC_PSO(2,IR), J28, 1, VAL_R8_ARR(2), IER )
                       IF ( IER .NE. 0 ) THEN
                            CALL ERR_LOG ( 2007, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                          'putting '//GVH__LC_PSO(2,IR)//' lcode' )
                            RETURN
                       END IF
 4280              CONTINUE 
                 ELSE IF ( C_LCD(J6) == 'EXP_CODE' ) THEN
                   CALL ERR_PASS   ( IUER, IER )
                   LEN_DATA = LEN(EXP_CODE)
                   CALL CLRCH ( EXP_CODE )
                   CALL GVH_GLCODE ( GVH_NS, C_LCD(J6), 1, 1, LEN_DATA, &
     &                               DIMS(1), DIMS(2), EXP_CODE, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 2008, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                      'an attempt to get data of lcode '//C_LCD(J6)// &
     &                      ' from the input database' )
                        RETURN 
                    END IF
                    CALL TRAN   ( 12, EXP_CODE, EXP_CODE )
!
                    CALL ERR_PASS   ( IUER, IER )
                    CALL GVH_PLCODE ( GVH, GVH__LC_PSO(1,IR), 1, 1, EXP_CODE, IER )
                    IF ( IER .NE. 0 ) THEN
                         CALL ERR_LOG ( 2009, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                       'putting '//GVH__LC_PSO(1,IR)//' lcode' )
                         RETURN
                    END IF
                 ELSE IF ( C_LCD(J6) == 'APBYFRQ1' ) THEN
                   CALL ERR_PASS   ( IUER, IER )
                   CALL GVH_GLCODE ( GVH_NS, 'NUM_CHBN', 1, 1, 8, DIMS(1), DIMS(2), NUM_CHNS, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 2010, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                      'an attempt to get data of lcode '//C_LCD(J6)// &
     &                      ' from the input database' )
                        RETURN 
                   END IF
                   DO 4290 J29=1,NUMOBS
                      LEN_DATA = 8*2*NUM_CHNS(1)
                      CALL ERR_PASS   ( IUER, IER )
                      CALL GVH_GLCODE ( GVH_NS, C_LCD(J6), J29, 1, LEN_DATA, &
     &                                  DIMS(1), DIMS(2), VAL_R8_ARR, IER )
                      IF ( IER .NE. 0 ) THEN
                           CALL ERR_LOG ( 2011, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                          'an attempt to get data of lcode '//C_LCD(J6)// &
     &                          ' from the input database' )
                           RETURN 
                      END IF
                      DO 4300 J30=1,NUM_CHNS(1)
                         UV_CHN(1,J30) = VAL_R8_ARR(J30)*DCOS ( VAL_R8_ARR(J30+NUM_CHNS(1)-1)*DEG__TO__RAD )
                         UV_CHN(2,J30) = VAL_R8_ARR(J30)*DSIN ( VAL_R8_ARR(J30+NUM_CHNS(1)-1)*DEG__TO__RAD )
 4300                 CONTINUE 

                      CALL ERR_PASS   ( IUER, IER )
                      CALL GVH_PLCODE ( GVH, GVH__LC_PSO(1,IR), J29, 1, UV_CHN, IER )
                      IF ( IER .NE. 0 ) THEN
                           CALL ERR_LOG ( 2012, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                         'putting '//GVH__LC_PSO(1,IR)//' lcode' )
                           RETURN
                      END IF
 4290              CONTINUE 
                 ELSE IF ( C_LCD(J6) == 'APBYFRQ2' ) THEN
                   CALL GVH_GLCODE ( GVH_NS, 'NUM_CHBN', 1, 1, 8, DIMS(1), DIMS(2), &
     &                               NUM_CHNS, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 2013, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                      'an attempt to get data of lcode '//C_LCD(J6)// &
     &                      ' from the input database' )
                        RETURN 
                   END IF
                   DO 4310 J31=1,NUMOBS
                      LEN_DATA = 8*2*NUM_CHNS(2)
                      CALL ERR_PASS   ( IUER, IER )
                      CALL GVH_GLCODE ( GVH_NS, C_LCD(J6), J31, 1, LEN_DATA, &
     &                                  DIMS(1), DIMS(2), VAL_R8_ARR, IER )
                      IF ( IER .NE. 0 ) THEN
                           CALL ERR_LOG ( 2014, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                          'an attempt to get data of lcode '//C_LCD(J6)// &
     &                          ' from the input database' )
                           RETURN 
                      END IF
                      DO 4320 J32=1,NUM_CHNS(2)
                         UV_CHN(1,J32) = VAL_R8_ARR(J32)*DCOS ( VAL_R8_ARR(J32+NUM_CHNS(2)-1)*DEG__TO__RAD )
                         UV_CHN(2,J32) = VAL_R8_ARR(J32)*DSIN ( VAL_R8_ARR(J32+NUM_CHNS(2)-1)*DEG__TO__RAD )
 4320                 CONTINUE 
!
                      CALL ERR_PASS   ( IUER, IER )
                      CALL GVH_PLCODE ( GVH, GVH__LC_PSO(1,IR), J31, 1, UV_CHN, IER )
                      IF ( IER .NE. 0 ) THEN
                           CALL ERR_LOG ( 2015, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                         'putting '//GVH__LC_PSO(1,IR)//' lcode' )
                           RETURN
                      END IF
 4310              CONTINUE 
                 ELSE IF ( C_LCD(J6) == 'UNPHASCL' ) THEN
                   LEN_DATA = 8*2*2
                   DO 4330 J33=1,NUMOBS
                      CALL ERR_PASS   ( IUER, IER )
                      CALL GVH_GLCODE ( GVH_NS, C_LCD(J6), J33, 1, LEN_DATA, &
     &                                  DIMS(1), DIMS(2), VAL_R8_ARR, IER )
                      IF ( IER .NE. 0 ) THEN
                           CALL ERR_LOG ( 2016, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                         'getting '//C_LCD(J6)//' lcode' )
                           RETURN
                      END IF
!
                      CALL ERR_PASS   ( IUER, IER )
                      CALL GVH_PLCODE ( GVH, GVH__LC_PSO(1,IR), J33, 1, VAL_R8_ARR, IER )
                      IF ( IER .NE. 0 ) THEN
                           CALL ERR_LOG ( 2017, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                         'putting '//GVH__LC_PSO(1,IR)//' lcode' )
                           RETURN
                      END IF
 4330              CONTINUE 
                 ELSE IF ( C_LCD(J6) == 'SCANNAME' ) THEN
                   L_SCA = 0
                   DO 4340 J34=1,NUMOBS
                      CALL CLRCH ( SCA_NAME )
                      LEN_DATA = LEN(SCA_NAME)
                      CALL ERR_PASS   ( IUER, IER )
                      CALL GVH_GLCODE ( GVH_NS, C_LCD(J6), J34, 1, LEN_DATA, &
     &                                  DIMS(1), DIMS(2), SCA_NAME, IER )
                      IF ( IER .NE. 0 ) THEN
                           CALL ERR_LOG ( 2018, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                         'getting '//C_LCD(J6)//' lcode' )
                           RETURN
                      END IF
!
                      CALL ERR_PASS   ( IUER, IER )
                      CALL GVH_GLCODE ( GVH_NS, 'SOU_IND ', J34, 1, LEN_DATA, &
     &                                  DIMS(1), DIMS(2), IND_SOU, IER )
                      IF ( IER .NE. 0 ) THEN
                           CALL ERR_LOG ( 2019, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                         'getting '//C_LCD(J6)//' lcode' )
                           RETURN
                      END IF
                      IF ( SCA_NAME(5:5) == '/' ) THEN
                           SCA_NAME = SCA_NAME(1:4)//SCA_NAME(6:7)//SCA_NAME(9:10)//'_'// &
     &                                SCA_NAME(12:13)//SCA_NAME(15:16)//SCA_NAME(18:19)
                           IF ( L_SCA > 0 ) THEN
                                DO 4350 J35=1,8
                                   IS = LTM_DIF ( 0, L_SCA, C_SCA, SCA_NAME )
                                   IF ( IS > 0 ) THEN
!
! ------------------------------------- This scan name was already used. Check if the source name
! ------------------------------------- it is associated with
!
                                        IF ( C_SOU(IND_SOU) .NE. C_SCA_SOU(IS) ) THEN
!
! ------------------------------------------ Aga! This scan name was used for another source
! ------------------------------------------ Update the last character in the scan name and try again
!
                                             SCA_NAME(16:16) = CHAR ( ICHAR('a') + J35 -1 )
                                        END IF
                                      ELSE 
!
! ------------------------------------- The scan name was not used
!
                                        GOTO 8350
                                    END IF
 4350                           CONTINUE 
 8350                           CONTINUE 
                           END IF
                      END IF
                      IF ( L_SCA > 0 ) THEN
                           IF ( LTM_DIF ( 0, L_SCA, C_SCA, SCA_NAME ) > 0 ) THEN
!
! ----------------------------- New scan
!
                                L_SCA = L_SCA + 1
                                C_SCA(L_SCA) = SCA_NAME
                                C_SCA_SOU(L_SCA) = C_SOU(IND_SOU)
                           END IF
                      END IF
!
                      CALL ERR_PASS   ( IUER, IER )
                      CALL GVH_PLCODE ( GVH, GVH__LC_PSO(1,IR), J34, 1, SCA_NAME, IER )
                      IF ( IER .NE. 0 ) THEN
                           CALL ERR_LOG ( 2020, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                         'putting '//GVH__LC_PSO(1,IR)//' lcode' )
                           RETURN
                      END IF
 4340              CONTINUE 
                 ELSE IF ( C_LCD(J6) == 'CLK_INTV' ) THEN
                   LEN_DATA = SIZEOF(CLO_INTR)
                   CALL ERR_PASS   ( IUER, IER )
                   CALL GVH_GLCODE ( GVH_NS, C_LCD(J6), 1, 1, LEN_DATA, &
     &                               DIMS(1), DIMS(2), CLO_INTR, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 2021, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                      'geting '//C_LCD(J6)//' lcode' )
                        RETURN
                   END IF
                   CLO_INTR(1:GVF_DB__MSLT) = CLO_INTR(1)*3600.0D0
!
                   CALL ERR_PASS   ( IUER, IER )
                   CALL GVH_PLCODE ( GVH, GVH__LC_PSO(1,IR), 1, 1, CLO_INTR, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 2022, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                      'putting '//GVH__LC_PSO(1,IR)//' lcode' )
                        RETURN
                   END IF
                 ELSE IF ( C_LCD(J6) == 'ATM_INTV' ) THEN
                   LEN_DATA = SIZEOF(ATM_INTR)
                   CALL ERR_PASS   ( IUER, IER )
                   CALL GVH_GLCODE ( GVH_NS, C_LCD(J6), 1, 1, LEN_DATA, &
     &                               DIMS(1), DIMS(2), ATM_INTR, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 2023, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                      'geting '//C_LCD(J6)//' lcode' )
                        RETURN
                   END IF
                   ATM_INTR(1:GVF_DB__MSLT) = ATM_INTR(1)*3600.0D0
!
                   CALL ERR_PASS   ( IUER, IER )
                   CALL GVH_PLCODE ( GVH, GVH__LC_PSO(1,IR), 1, 1, CLO_INTR, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 2024, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                      'putting '//GVH__LC_PSO(1,IR)//' lcode' )
                        RETURN
                   END IF
                 ELSE IF ( C_LCD(J6) == 'RFREQ1  ' ) THEN
                   CALL GVH_GLCODE ( GVH_NS, 'NUM_CHBN', 1, 1, 8, DIMS(1), DIMS(2), NUM_CHNS, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 2025, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                      'an attempt to get data of lcode '//C_LCD(J6)// &
     &                      ' from the input database' )
                        RETURN 
                   END IF
                   DO 4360 J36=1,NUMOBS
                      NUSEDCHN = 0
                      IND_CHN1 = 0
                      IND_CHN2 = 0
                      CALL GVH_GLCODE ( GVH_NS, C_LCD(J6), J36, 1, 8*NUM_CHNS(1), &
     &                                  DIMS(1), DIMS(2), SKYFRQCH, IER )
                      IF ( IER .NE. 0 ) THEN
                           CALL ERR_LOG ( 2026, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                         'an attempt to get data of lcode '//C_LCD(J6)// &
     &                         ' from the input database' )
                           RETURN 
                      END IF
                      IF ( NUMBAND > 1 ) THEN
                           CALL GVH_GLCODE ( GVH_NS, 'RFREQ2  ', J36, 1, 8*NUM_CHNS(2), &
     &                                       DIMS(1), DIMS(2), SKYFRQCH(NUM_CHNS(1)+1), IER )
                           IF ( IER .NE. 0 ) THEN
                                CALL ERR_LOG ( 2027, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                              'an attempt to get data of lcode RFREQ2  '// &
     &                              ' from the input database' )
                                RETURN 
                           END IF
                      END IF
                      DO 4370 J37=1,NUM_CHNS(1)
                         IF ( IS_R8_NAN ( SKYFRQCH(J37) ) ) THEN
                              SKYFRQCH(J37) = 0.D0
                         END IF
                         IF ( SKYFRQCH(J37) > 0.01 ) THEN
                              NUSEDCHN(1) = NUSEDCHN(1) + 1
                              IND_CHN1(J37) = J37
                              FRQ_MIN = MIN ( FRQ_MIN, SKYFRQCH(J37) )
                              FRQ_MAX = MAX ( FRQ_MAX, SKYFRQCH(J37) )
                         END IF
 4370                 CONTINUE 
                      IF ( NUMBAND > 1 ) THEN
                           DO 4380 J38=1,NUM_CHNS(2)
                              IF ( IS_R8_NAN ( SKYFRQCH(NUM_CHNS(1)+J38) ) ) THEN
                                   SKYFRQCH(NUM_CHNS(1)+J38) = 0.0D0
                              END IF
                              IF ( SKYFRQCH(NUM_CHNS(1)+J38) > 0.01 ) THEN
                                   NUSEDCHN(2) = NUSEDCHN(2) + 1
                                   IND_CHN2(J38) = J38
                                   FRQ_MIN = MIN ( FRQ_MIN, SKYFRQCH(NUM_CHNS(1)+J38) )
                                   FRQ_MAX = MAX ( FRQ_MAX, SKYFRQCH(NUM_CHNS(1)+J38) )
                              END IF
 4380                     CONTINUE 
                      END IF
!
                      CALL ERR_PASS   ( IUER, IER )
                      CALL GVH_PLCODE ( GVH, C_LCD(J6), J36, 1, SKYFRQCH, IER )
                      IF ( IER .NE. 0 ) THEN
                           CALL ERR_LOG ( 2028, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                         'putting '//C_LCD(J6)//' lcode' )
                           RETURN
                      END IF
!
                      CALL ERR_PASS   ( IUER, IER )
                      CALL GVH_PLCODE ( GVH, GVH__LC_PSO(1,IR), J36, 1, IND_CHN1, IER )
                      IF ( IER .NE. 0 ) THEN
                           CALL ERR_LOG ( 2029, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                         'putting '//GVH__LC_PSO(1,IR)//' lcode' )
                           RETURN
                      END IF
!
                      IF ( NUMBAND > 1 ) THEN
                           CALL ERR_PASS   ( IUER, IER )
                           CALL GVH_PLCODE ( GVH, 'IND_CHN2', J36, 1, IND_CHN2, IER )
                           IF ( IER .NE. 0 ) THEN
                                CALL ERR_LOG ( 2030, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                              'putting "'//GVH__LC_PSO(2,IR)//'" lcode' )
                               RETURN
                           END IF
                      END IF
!
                      IF ( .NOT. FL_NUSEDCHN ) THEN
                           CALL ERR_PASS   ( IUER, IER )
                           CALL GVH_PLCODE ( GVH, GVH__LC_PSO(2,IR), J36, 1, NUSEDCHN, IER )
                           IF ( IER .NE. 0 ) THEN
                                CALL ERR_LOG ( 2031, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                              'putting '//GVH__LC_PSO(2,IR)//' lcode' )
                                RETURN
                           END IF
                      END IF
 4360              CONTINUE 
                 ELSE IF ( C_LCD(J6) == 'RFREQ2  ' ) THEN
                   CALL GVH_GLCODE ( GVH_NS, 'NUM_CHBN', 1, 1, 8, DIMS(1), DIMS(2), NUM_CHNS, IER )
                   IF ( IER .NE. 0 ) THEN
                        CALL ERR_LOG ( 2032, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                      'an attempt to get data of lcode NUM_CHBN '//&
     &                      ' from the input database' )
                        RETURN 
                   END IF
                   DO 4390 J39=1,NUMOBS
                      CALL GVH_GLCODE ( GVH_NS, C_LCD(J6), J39, 1, 8*NUM_CHNS(2), &
     &                                  DIMS(1), DIMS(2), SKYFRQCH(NUM_CHNS(1)+1), IER )
                      IF ( IER .NE. 0 ) THEN
                           CALL ERR_LOG ( 2033, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                         'an attempt to get data of lcode '//C_LCD(J6)// &
     &                         ' from the input database' )
                           RETURN 
                      END IF
!
                      CALL ERR_PASS   ( IUER, IER )
                      CALL GVH_PLCODE ( GVH, C_LCD(J6), J39, 1, SKYFRQCH(NUM_CHNS(1)+1), IER )
                      IF ( IER .NE. 0 ) THEN
                           CALL ERR_LOG ( 2034, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                         'putting '//C_LCD(J6)//' lcode' )
                           RETURN
                      END IF
 4390              CONTINUE 
                 ELSE IF ( C_LCD(J6) == 'QUALCODE'  .AND. &
     &                     LTM_DIF ( 0, L_LCD, C_LCD, 'DELUFLAG' ) .LE. 0 ) THEN
!
! ---------------- We update QUALCODE flag if unweighing flag was not specified
! ---------------- (this happens if the database was not processed with Solve)
! ---------------- Otherwise QIALCODE was already updated
!
                   DO 4400 J40=1,NUMOBS
                      CALL ERR_PASS   ( IUER, IER )
                      CALL GVH_GLCODE ( GVH_NS, 'QUALCODE', J40, 1, LEN_DATA, &
     &                                  DIMS(1), DIMS(2), STR, IER )
                      IF ( IER .NE. 0 ) THEN
                           CALL ERR_LOG ( 2035, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                         'an attempt to get data of lcode QUALCODE'// &
     &                         ' from the input database' )
                           RETURN 
                      END IF
!
                      IF ( DIMS(1) == 1 ) THEN
                           QUALCODE(1) = STR(1:1)
                           QUALCODE(2) = STR(2:2)
                         ELSE
                           QUALCODE(1) = STR(1:1)
                           QUALCODE(2) = STR(3:3)
                      END IF
                      USER_SUP = 0
                      IF ( QUALCODE(1) == ' ' .OR. QUALCODE(1) == '-' ) THEN
                           QUALCODE(1) = '0'
                      END IF
                      IF ( QUALCODE(2) == ' ' .OR. QUALCODE(2) == '-' ) THEN
                           QUALCODE(2) = '0'
                      END IF
!
                      CALL ERR_PASS   ( IUER, IER )
                      CALL GVH_PLCODE ( GVH, 'QUALCODE', J40, 1, QUALCODE, IER )
                      IF ( IER .NE. 0 ) THEN
                           CALL ERR_LOG ( 2036, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                         'putting "QUALCODE" lcode' )
                           RETURN
                      END IF
 4400              CONTINUE 
                 ELSE IF ( C_LCD(J6) == 'FRTYPFIT' ) THEN
!
! ---------------- We will treat it in the lew lcode section
!
                   CONTINUE 
              END IF
              GOTO 460
         END IF
!
! ------ Bypass an LCODE with "ignore" status
!
         IF ( GVH__TBI(IP) == GVH__IGN ) GOTO 460
!
! ------ Bypass mandatory code, since it has already been uploaded
!
         IF ( LTM_DIF ( 0, GVH__NMLCODE, GVH__MLCODE, C_LCD(J6) ) > 0 ) GOTO 460
!
         CALL ERR_PASS   ( IUER, IER )
         CALL GVH_INQ_LCODE ( GVH_NS, C_LCD(J6), DESCR, CLASS, TYP, DIMS, &
     &                        NUM_FIELDS, SEG_IND, LEN_REC, LEN_DATA, &
     &                        ADR_DATA, IER )
         IF ( IER .NE. 0 ) THEN
              CALL ERR_LOG ( 2037, IUER, 'GVH_FROM_NUSOLVE', 'Error in an attempt to '// &
     &            'execute INQ_LCODE for '//C_LCD(J6) )
              RETURN 
         END IF
         CALL GET_MEM ( INT8(LEN_DATA), ADR_DATA )
         IF ( ADR_DATA == 0 ) THEN
              CALL CLRCH ( STR ) 
              CALL IINCH ( LEN_DATA, STR )
              CALL ERR_LOG ( 2038, IUER, 'GVH_FROM_NUSOLVE', 'Error in an attempt to '// &
     &            'allocate '//TRIM(STR)//' bytes of dynamic memory' )
              RETURN 
         END IF
!
         IF ( CLASS == GVH__SES ) THEN
              CALL ERR_PASS   ( IUER, IER )
              CALL GVH_GLCODE ( GVH_NS, C_LCD(J6), 0, 0, LEN_DATA, &
     &                          DIMS(1), DIMS(2), %VAL(ADR_DATA), IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 2039, IUER, 'GVH_FROM_NUSOLVE', 'Error in an attempt to '// &
     &                 'get data of lcode '//C_LCD(J6)//' from the input database' )
                   RETURN 
              END IF
!
              CALL ERR_PASS   ( IUER, IER )
              CALL GVH_PLCODE ( GVH, C_LCD(J6), 0, 0, %VAL(ADR_DATA), IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 2040, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                 'an attempt to put data of lcode '//C_LCD(J6)// &
     &                 ' from the output database' )
                   RETURN 
              END IF
           ELSE IF ( CLASS == GVH__BAS ) THEN
              DO 4410 J41=1,NUMOBS
                 CALL ERR_PASS   ( IUER, IER )
                 CALL GVH_GLCODE ( GVH_NS, C_LCD(J6), J41, 0, LEN_DATA, &
     &                             DIMS(1), DIMS(2), %VAL(ADR_DATA), IER )
                 IF ( IER .NE. 0 ) THEN
                      CALL ERR_LOG ( 2041, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                    'an attempt to get data of lcode '//C_LCD(J6)// &
     &                    ' from the input database' )
                      RETURN 
                 END IF
!
                 CALL GVH_PLCODE ( GVH, C_LCD(J6), J41, 0, %VAL(ADR_DATA), IER )
                 IF ( IER .NE. 0 ) THEN
                      CALL ERR_LOG ( 2042, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                    'an attempt to get data of lcode '//C_LCD(J6)// &
     &                    ' to the output database' )
                      RETURN 
                 END IF
 4410         CONTINUE 
           ELSE IF ( CLASS == GVH__STA ) THEN
              DO 4420 J42=1,NUMOBS
                 CALL GVH_GLCODE ( GVH_NS, C_LCD(J6), J42, 1, LEN_DATA, &
     &                             DIMS(1), DIMS(2), %VAL(ADR_DATA), IER )
                 IF ( IER .NE. 0 ) THEN
                      CALL ERR_LOG ( 2043, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                    'an attempt to get data of lcode '//C_LCD(J6)// &
     &                    ' from the output database' )
                      RETURN 
                 END IF
!
                 CALL ERR_PASS   ( IUER, IER )
                 CALL GVH_PLCODE ( GVH, C_LCD(J6), J42, 1, %VAL(ADR_DATA), IER )
                 IF ( IER .NE. 0 ) THEN
                      CALL ERR_LOG ( 2044, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                    'an attempt to put data of lcode '//C_LCD(J6)// &
     &                    ' to the output database' )
                      RETURN 
                 END IF
!
                 CALL GVH_GLCODE ( GVH_NS, C_LCD(J6), J42, 2, LEN_DATA, &
     &                             DIMS(1), DIMS(2), %VAL(ADR_DATA), IER )
                 IF ( IER .NE. 0 ) THEN
                      CALL ERR_LOG ( 2045, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                    'an attempt to get data of lcode '//C_LCD(J6)// &
     &                    ' from the innput database' )
                      RETURN 
                 END IF
!
                 CALL ERR_PASS   ( IUER, IER )
                 CALL GVH_PLCODE ( GVH, C_LCD(J6), J42, 2, %VAL(ADR_DATA), IER )
                 IF ( IER .NE. 0 ) THEN
                      CALL ERR_LOG ( 2046, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                    'an attempt to put data of lcode '//C_LCD(J6)// &
     &                    ' to the output database' )
                      RETURN 
                 END IF
 4420         CONTINUE 
           ELSE IF ( CLASS == GVH__SCA ) THEN
              DO 4430 J43=1,NUMOBS
                 CALL ERR_PASS   ( IUER, IER )
                 CALL GVH_GLCODE ( GVH_NS, C_LCD(J6), J43, 0, LEN_DATA, &
     &                             DIMS(1), DIMS(2), %VAL(ADR_DATA), IER )
                 IF ( IER .NE. 0 ) THEN
                      CALL ERR_LOG ( 2047, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                    'an attempt to put data of lcode '//C_LCD(J6)// &
     &                    ' to the output database' )
                      RETURN 
                 END IF
!
                 CALL ERR_PASS   ( IUER, IER )
                 CALL GVH_PLCODE ( GVH, C_LCD(J6), J43, 0, %VAL(ADR_DATA), IER )
                 IF ( IER .NE. 0 ) THEN
                      CALL ERR_LOG ( 2048, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                    'an attempt to put data of lcode '//C_LCD(J6)// &
     &                    ' to the output database' )
                      RETURN 
                 END IF
 4430         CONTINUE 
         END IF 
         CALL FREE_MEM ( ADR_DATA )
 460  CONTINUE 
!
      DO 4440 J44=1,GVH__NEW_NUS
         IF ( GVH__LC_NEW(J44) == 'REC_MODE' .AND. &
     &        LTM_DIF ( 0, L_LCD, C_LCD, GVH__LC_NEW(J44) ) .LE. 0 ) THEN
!
              CALL CLRCH ( STR )
              STR = '????????'
              CALL ERR_PASS   ( IUER, IER )
              CALL GVH_PLCODE ( GVH, GVH__LC_NEW(J44), 1, 0, %VAL(LOC(STR)), IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 2049, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                 'an attempt to put data of lcode '//C_LCD(J6)// &
     &                 ' to the output database' )
                  RETURN 
              END IF
           ELSE IF ( GVH__LC_NEW(J44) == 'EXPSERNO' ) THEN
              CALL ERR_PASS   ( IUER, IER )
              CALL GVH_PLCODE ( GVH, GVH__LC_NEW(J44), 1, 0, 0, IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 2050, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                 'an attempt to put data of lcode '//C_LCD(J6)// &
     &                 ' to the output database' )
                  RETURN 
              END IF
           ELSE IF ( GVH__LC_NEW(J44) == 'EDIT_STS' ) THEN
              EDIT_STS = 0
              IF ( FL_CLK ) THEN
                   EDIT_STS = IBSET ( EDIT_STS, GX__DTP_PAR )
                   IF ( NUMBAND > 1 ) THEN
                        EDIT_STS = IBSET ( EDIT_STS, GR__DTP_PAR  )
                        EDIT_STS = IBSET ( EDIT_STS, GXS__DTP_PAR )
                   END IF
              END IF
!
              CALL ERR_PASS   ( IUER, IER )
              CALL GVH_PLCODE ( GVH, 'EDIT_STS', 1, 1, EDIT_STS, IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 2051, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                 'putting "EDIT_STS" lcode' )
                   RETURN
              END IF
           ELSE IF ( GVH__LC_NEW(J44) == 'TH_PROG ' ) THEN
              CALL ERR_PASS   ( IUER, IER )
              CALL CLRCH ( STR )
              STR = 'CALC'
              CALL GVH_PLCODE ( GVH, 'TH_PROG ', 1, 1, STR, IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 2052, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                 'putting "TH_PROG" lcode' )
                   RETURN
              END IF
           ELSE IF ( GVH__LC_NEW(J44) == 'DGCL_EST' ) THEN
              CALL ERR_PASS   ( IUER, IER )
              CALL GVH_PLCODE ( GVH, 'DGCL_EST', 1, 1, 2, IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 2053, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                 'putting "DGCL_EST" lcode' )
                   RETURN
              END IF
           ELSE IF ( GVH__LC_NEW(J44) == 'DATYP   ' ) THEN
               IF ( LTM_DIF ( 0, L_LCD, C_LCD, 'FOURFVER' ) .GT. 0 ) THEN
                    IF ( NUMBAND == 1 .AND. FRQ_MIN .LE. 4.0D3 .AND. FRQ_MAX .GE. 8.0D3 ) THEN
                         DATYP_I2 = GR__DTP_PAR
                       ELSE IF ( NUMBAND == 1 ) THEN
                         DATYP_I2 = GX__DTP_PAR
                       ELSE
                         DATYP_I2 = GXS__DTP_PAR
                    END IF
                 ELSE
                   DATYP_I2 = GR__DTP_PAR
              END IF
!
              CALL ERR_PASS   ( IUER, IER )
              CALL GVH_PLCODE ( GVH, 'DATYP   ', 1, 1, DATYP_I2, IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 2054, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                 'putting "DATYP" lcode' )
                   RETURN
              END IF
           ELSE IF ( GVH__LC_NEW(J44) == 'SUPMET  ' ) THEN
              SUPMET_I2 = SUPMET__META
!
              CALL ERR_PASS   ( IUER, IER )
              CALL GVH_PLCODE ( GVH, 'SUPMET  ', 1, 1, SUPMET_I2, IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 2055, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                 'putting "SUPMET" lcode' )
                   RETURN
              END IF
           ELSE IF ( GVH__LC_NEW(J44) == 'UTC_MTAI' ) THEN
              CALL ERR_PASS ( IUER, IER )
              CALL NERS_INIT ( NERS_CONFIG, NERS, -1.0D0, -1.0D0, IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 2056, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                 'an attempt to initialize NERS object' )
                   RETURN
              END IF
!
              CALL ERR_PASS ( IUER, IER )
              CALL NERS_LOAD ( NERS, IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 2057, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                 'an attempt to retrieve NERS forecast parameters '// &
     &                 'form the remote server' )
                   RETURN
              END IF
!
              UTC_TAG = (MJD_OBS(1) - J2000__MJD)*86440.0D0 + UTC_OBS(1)
              CALL ERR_PASS ( IUER, IER )
              CALL NERS_GET_UTCMTAI ( NERS, UTC_TAG, UTC_M_TAI, IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 2058, IUER, 'GVH_FROM_NUSOLVE', 'Error in getting '// &
     &                 'UTC minus TAI' )
                   RETURN
              END IF
!
              CALL ERR_PASS   ( IUER, IER )
              CALL GVH_PLCODE ( GVH, 'UTC_MTAI', 1, 1, UTC_M_TAI, IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 2059, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                 'putting "UTC_MTAI" lcode' )
                   RETURN
              END IF
           ELSE IF ( GVH__LC_NEW(J44) == 'TEC_STS ' ) THEN
              TEC_STS_STA = 0
              DO 4450 J45=1,NUMSTA
                 IF ( NUMBAND > 1 ) THEN
                      TEC_STS_STA(J45) = IBSET( TEC_STS_STA(J45), 0 )
                      TEC_STS_STA(J45) = IBSET( TEC_STS_STA(J45), 1 )
                 END IF
 4450         CONTINUE 
!
              CALL ERR_PASS   ( IUER, IER )
              CALL GVH_PLCODE ( GVH, 'TEC_STS ', 1, 1, TEC_STS_STA, IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 2060, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                 'putting "TEC_STS" lcode' )
                   RETURN
              END IF
           ELSE IF ( GVH__LC_NEW(J44) == 'AUTO_SUP ' ) THEN
              DO 4460 J46=1,NUMOBS
                 CALL ERR_PASS   ( IUER, IER )
                 CALL GVH_GLCODE ( GVH_NS, 'QUALCODE', J46, 1, LEN_DATA, &
     &                             DIMS(1), DIMS(2), STR, IER )
                 IF ( IER .NE. 0 ) THEN
                      CALL ERR_LOG ( 2061, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                    'an attempt to get data of lcode QUALCODE'// &
     &                    ' from the input database' )
                      RETURN 
                 END IF
!
                 IF ( DIMS(1) == 1 ) THEN
                      QUALCODE(1) = STR(1:1)
                      QUALCODE(2) = STR(2:2)
                    ELSE
                      QUALCODE(1) = STR(1:1)
                      QUALCODE(2) = STR(3:3)
                 END IF
                 IF ( QUALCODE(1) == ' ' .OR. QUALCODE(1) == '-' ) THEN
                      QUALCODE(1) = '0'
                 END IF
                 IF ( QUALCODE(2) == ' ' .OR. QUALCODE(2) == '-' ) THEN
                      QUALCODE(2) = '0'
                 END IF
!
                 AUTO_SUP(J46) = 0
                 AUTO_SUP(J46) = IBSET ( AUTO_SUP(J46), INT4(INIT__SPS) )
!
                 IF ( QUALCODE(1) == '0' .OR.    &     
     &                     ( ICHAR(QUALCODE(1)) .GE. ICHAR('A') .AND.   &     
     &                       ICHAR(QUALCODE(1)) .LE. ICHAR('F')       ) ) THEN
!
! ------------------- No detection at X-band
!
                      AUTO_SUP(J46) = IBSET ( AUTO_SUP(J46), INT4(NOFX__SPS) )
                    ELSE IF ( ( ICHAR(QUALCODE(1)) .GE. ICHAR('1') .AND.   &     
     &                          ICHAR(QUALCODE(1)) .LE. ICHAR('5')       ) ) THEN
!
! ------------------- Bad quality code at X-band
!
                      AUTO_SUP(J46) = IBSET ( AUTO_SUP(J46), INT4(BQCX__SPS) )
                 END IF
!
                 IF ( NUMBAND > 1 ) THEN
                      IF ( QUALCODE(2) == '0' .OR.    &     
     &                     ( ICHAR(QUALCODE(2)) .GE. ICHAR('A') .AND.   &     
     &                       ICHAR(QUALCODE(2)) .LE. ICHAR('F')       ) ) THEN
!
! ------------------------ No detection at S-band
!
                           AUTO_SUP(J46) = IBSET ( AUTO_SUP(J46), INT4(NOFS__SPS) )
                         ELSE IF ( ( ICHAR(QUALCODE(2)) .GE. ICHAR('1') .AND.   &     
     &                               ICHAR(QUALCODE(2)) .LE. ICHAR('5')       ) ) THEN
!
! ------------------------ Bad quality code at S-band
!
                           AUTO_SUP(J46) = IBSET ( AUTO_SUP(J46), INT4(BQCS__SPS) )
                      END IF
                 END IF
!
                 CALL ERR_PASS   ( IUER, IER )
                 CALL GVH_PLCODE ( GVH, 'AUTO_SUP', J46, 1, AUTO_SUP(J46), IER )
                 IF ( IER .NE. 0 ) THEN
                      CALL ERR_LOG ( 2062, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                    'putting "AUTO_SUP" lcode' )
                      RETURN
                 END IF
 4460         CONTINUE 
           ELSE IF ( GVH__LC_NEW(J44) == 'NUM_CLRF' .AND. FL_CLK ) THEN
              CALL ERR_PASS   ( IUER, IER )
              CALL GVH_PLCODE ( GVH, 'NUM_CLRF', 1, 1, NUM_CLRF, IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 2063, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                 'putting "NUM_CLRF" lcode' )
                   RETURN
              END IF
           ELSE IF ( GVH__LC_NEW(J44) == 'FRTYPFIT' .AND. &
     &               LTM_DIF ( 0, L_LCD, C_LCD, GVH__LC_NEW(J44) ) > 0 ) THEN
!
! ----------- FRTYPFIT already present in the database
!
              CALL GVH_GLCODE ( GVH_NS, GVH__LC_NEW(J44), 1, 1, LEN_DATA, &
     &                          DIMS(1), DIMS(2), FRTYPFIT, IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 2064, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                 'an attempt to get data of lcode '//GVH__LC_NEW(J44)// &
     &                 ' from the innput database' )
                   RETURN 
              END IF
!
              CALL ERR_PASS   ( IUER, IER )
              CALL GVH_PLCODE ( GVH, 'FRTYPFIT', 1, 1, FRTYPFIT, IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 2065, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                  'putting "FRTYPFIT" lcode' )
                   RETURN
              END IF
           ELSE IF ( GVH__LC_NEW(J44) == 'FRTYPFIT' .AND. &
     &               LTM_DIF ( 0, L_LCD, C_LCD, GVH__LC_NEW(J44) ) .LE. 0 ) THEN
              FRTYPFIT = UNDF__FTP
              IF ( LTM_DIF ( 0, L_LCD, C_LCD, 'PIMA_CNT' ) .GT. 0 ) THEN
                   FRTYPFIT = PHS_GRP_PIMA__FTP
                 ELSE IF ( LTM_DIF ( 0, L_LCD, C_LCD, 'FOURFVER' ) .GT. 0 ) THEN
                   IF ( NUMBAND == 1 .AND. FRQ_MIN .LE. 4.0D3 .AND. FRQ_MAX .GE. 8.0D3 ) THEN
                        FRTYPFIT = PHS_GRP_DTEC_HOPS__FTP
                      ELSE 
                        FRTYPFIT = PHS_GRP_HOPS__FTP
                   END IF
              END IF
!
              CALL ERR_PASS   ( IUER, IER )
              CALL GVH_PLCODE ( GVH, 'FRTYPFIT', 1, 1, FRTYPFIT, IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 2066, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                  'putting "FRTYPFIT" lcode' )
                   RETURN
              END IF
           ELSE IF ( GVH__LC_NEW(J44) == 'DB_VERS ' .AND. &
     &               LTM_DIF ( 0, L_LCD, C_LCD, GVH__LC_NEW(J44) ) .LE. 0 ) THEN
              IF ( FILDB(1:1) == '@' .AND. &
     &             ICHAR(FILDB(2:2)) .GE. ICHAR('a') .AND. &
     &             ICHAR(FILDB(2:2)) .LE. ICHAR('z')       ) THEN
!
! ---------------- Generate the output database name from the start data and the 
! ---------------- suffix that followed up @ in the input database name
!
                   VAL_I2_ARR(1) = 1
                 ELSE
                   READ ( UNIT=FILDB(13:15), FMT='(I3)' ) VAL_I2_ARR(1)
              END IF
!
              CALL ERR_PASS   ( IUER, IER )
              CALL GVH_PLCODE ( GVH, 'DB_VERS ', 1, 1, VAL_I2_ARR(1), IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 2067, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                  'putting "DB_VERS" lcode' )
                   RETURN
              END IF
         END IF
 4440 CONTINUE 
!
      IF ( LTM_DIF ( 0, L_LCD, C_LCD, 'NSAMPLS1' ) .LE. 0 ) THEN
           CALL ERR_PASS   ( IUER, IER )
           CALL GVH_GLCODE ( GVH_NS, 'NUM_CHBN', 1, 1, 8, DIMS(1), DIMS(2), NUM_CHNS, IER )
           IF ( IER .NE. 0 ) THEN
                CALL ERR_LOG ( 2068, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &              'an attempt to get data of lcode NUM_CHBN '// &
     &              ' from the input database' )
                RETURN 
           END IF
!
           DO 4470 J47=1,NUMOBS
              CALL ERR_PASS   ( IUER, IER )
              CALL GVH_GLCODE ( GVH_NS, 'NUM_AP1 ', J47, 1, 2*2*NUM_CHNS(1), &
     &                          DIMS(1), DIMS(2), VAL_I2_ARR, IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 2069, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                 'an attempt to get data of lcode NUM_AP1 '// &
     &                 ' from the input database' )
                   RETURN 
              END IF
!
              CALL ERR_PASS   ( IUER, IER )
              CALL GVH_GLCODE ( GVH_NS, 'SAMPLRAT', J47, 1, 4*8, DIMS(1), DIMS(2), &
     &                          SAMPLRAT, IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 2070, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                 'an attempt to get data of lcode SAMPLRAT '// &
     &                 ' from the input database' )
                   RETURN 
              END IF
!
              DO 4480 J48=1,NUM_CHNS(1)
                 VAL_R8_ARR(J48) = SAMPLRAT(1)*VAL_I2_ARR(J48)
                 VAL_R8_ARR(J48+NUM_CHNS(1)) = SAMPLRAT(1)*VAL_I2_ARR(J48+NUM_CHNS(1))
 4480         CONTINUE 
!
              CALL ERR_PASS   ( IUER, IER )
              CALL GVH_PLCODE ( GVH, 'NUM_SAM1', J47, 1, VAL_R8_ARR, IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 2071, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                  'putting "NUM_SAM1" lcode' )
                   RETURN
              END IF
 4470      CONTINUE 
      END IF
!
      IF ( NUMBAND > 1 .AND. LTM_DIF ( 0, L_LCD, C_LCD, 'NSAMPLS2' ) .LE. 0 ) THEN
           CALL ERR_PASS   ( IUER, IER )
           CALL GVH_GLCODE ( GVH_NS, 'NUM_CHBN', 1, 1, 8, DIMS(1), DIMS(2), NUM_CHNS, IER )
           IF ( IER .NE. 0 ) THEN
                CALL ERR_LOG ( 2072, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &              'an attempt to get data of lcode NUM_CHBN '// &
     &              ' from the input database' )
                RETURN 
           END IF
!
           DO 4490 J49=1,NUMOBS
              CALL ERR_PASS   ( IUER, IER )
              CALL GVH_GLCODE ( GVH_NS, 'NUM_AP2 ', J49, 1, 2*2*NUM_CHNS(2), &
     &                          DIMS(1), DIMS(2), VAL_I2_ARR, IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 2073, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                 'an attempt to get data of lcode NUM_AP2 '// &
     &                 ' from the input database' )
                   RETURN 
              END IF
!
              CALL ERR_PASS   ( IUER, IER )
              CALL GVH_GLCODE ( GVH_NS, 'SAMPLRAT', J49, 1, 4*8, DIMS(1), DIMS(2), &
     &                          SAMPLRAT, IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 2074, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                 'an attempt to get data of lcode SAMPLRAT '// &
     &                 ' from the input database' )
                   RETURN 
              END IF
!
              DO 4500 J50=1,NUM_CHNS(2)
                 VAL_R8_ARR(J50) = SAMPLRAT(1)*VAL_I2_ARR(J50)
                 VAL_R8_ARR(J50+NUM_CHNS(2)) = SAMPLRAT(1)*VAL_I2_ARR(J50+NUM_CHNS(2))
 4500         CONTINUE 
!
              CALL ERR_PASS   ( IUER, IER )
              CALL GVH_PLCODE ( GVH, 'NUM_SAM2', J49, 1, VAL_R8_ARR, IER )
              IF ( IER .NE. 0 ) THEN
                   CALL ERR_LOG ( 2075, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &                  'putting "NUM_SAM2" lcode' )
                   RETURN
              END IF
 4490      CONTINUE 
      END IF
!
      IF ( .NOT. FL_REW ) THEN
!
! -------- Special case: no re-weighting parameters were found in the database.
! -------- Then we initialize these parameteres with zeroes
!
           IR = LTM_DIF ( 0, GVH__PRC_NUS, GVH__LC_NUS, 'ERROR_K ' )
           RWDELVAL = 0.0D0
           RWRATVAL = 0.0D0
!
           CALL ERR_PASS   ( IUER, IER )
           CALL GVH_PLCODE ( GVH, GVH__LC_PSO(1,IR), 1, 1, RWDELVAL, IER )
           IF ( IER .NE. 0 ) THEN
                CALL ERR_LOG ( 2076, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &              'an attempt to put data of lcode '//GVH__LC_PSO(1,IR)// &
     &              ' from the output database' )
                RETURN 
           END IF
!
           CALL ERR_PASS   ( IUER, IER )
           CALL GVH_PLCODE ( GVH, GVH__LC_PSO(2,IR), 1, 1, RWRATVAL, IER )
           IF ( IER .NE. 0 ) THEN
                CALL ERR_LOG ( 2077, IUER, 'GVH_FROM_NUSOLVE', 'Error in '// &
     &              'an attempt to put data of lcode '//GVH__LC_PSO(2,IR)// &
     &              ' from the output database' )
                RETURN 
           END IF
      END IF
!
! --- Copy text sections for each segment
!
      DO 4510 J51=1,GVH_NS%NSEG
         GVH%TEXT(J51) = GVH_NS%TEXT(J51)
 4510 CONTINUE
!
      IF ( FILDB(1:1) == '@' .AND. &
     &     ICHAR(FILDB(2:2)) .GE. ICHAR('a') .AND. &
     &     ICHAR(FILDB(2:2)) .LE. ICHAR('z')       ) THEN
!
! -------- Generate the output database name from the start data and the 
! -------- suffix that followed up @ in the input database name
!
           STR = MJDSEC_TO_DATE ( MJD_OBS(1), UTC_OBS(1), IER )
           FILDB = STR(1:4)//STR(6:7)//STR(9:10)//'_'//FILDB(2:2)//'_v001'
      END IF
!
      GVH%DB_VERS = 1
!
      GVH%OLD_FILENAME(GVH__FR1) = TRIM(GVF_DB_DIR)//'/'//FILDB(1:8)//'_'//FILDB(10:10)//'_'// &
     &                                  TRIM(EXP_CODE)//'_fr1_'//FILDB(12:15)//'.bgv'
      GVH%OLD_FILENAME(GVH__FR2) = TRIM(GVF_DB_DIR)//'/'//FILDB(1:8)//'_'//FILDB(10:10)//'_'// &
     &                                  TRIM(EXP_CODE)//'_fr2_'//FILDB(12:15)//'.bgv'
      GVH%OLD_FILENAME(GVH__CL1) = TRIM(GVF_DB_DIR)//'/'//FILDB(1:8)//'_'//FILDB(10:10)//'_'// &
     &                                  TRIM(EXP_CODE)//'_cl1_'//FILDB(12:15)//'.bgv'
      GVH%OLD_FILENAME(GVH__SL1) = TRIM(GVF_DB_DIR)//'/'//FILDB(1:8)//'_'//FILDB(10:10)//'_'// &
     &                                  TRIM(EXP_CODE)//'_sl1_'//FILDB(12:15)//'.bgv'
      L_ENV = GVH%NSEG 
      FILVER = FILDB(13:15)
      IF ( FILVER(1:2) == ' 0' ) FILVER = '  '//FILVER(3:3)
      IF ( FILVER(1:1) == '0'  ) FILVER = ' '//FILVER(2:3)
      CALL BLANK_TO_ZERO ( FILVER )
      BUF_ENV(GVH__FR1) = 'SYS MAN fr1 '//FILVER//' bgv '//FILDB(10:10)//' '//EXP_CODE
      BUF_ENV(GVH__FR2) = 'SYS MAN fr2 '//FILVER//' bgv '//FILDB(10:10)//' '//EXP_CODE
      BUF_ENV(GVH__CL1) = 'SYS MAN cl1 '//FILVER//' bgv '//FILDB(10:10)//' '//EXP_CODE
      BUF_ENV(GVH__SL1) = 'SYS MAN sl1 '//FILVER//' bgv '//FILDB(10:10)//' '//EXP_CODE
!
      CALL ERR_LOG ( 0, IUER )
      RETURN
      END  SUBROUTINE  GVH_FROM_NUSOLVE   !#!#
#ifdef GVH_STANDALONE
!
! ------------------------------------------------------------------------
!
      FUNCTION KBIT ( IARRAY, IBIT )
      IMPLICIT NONE
!
! 1.  KBIT PROGRAM SPECIFICATION
!
! 1.1 Test the specified bit of the specified array to see
!     whether it is set or not.  designed to complement SBIT,
!     which sets or resets bits identified in the same way.
!
! 1.2 REFERENCES:
!
! 2.  KBIT INTERFACE
!
! 2.1 Parameter File
!
! 2.2 INPUT Variables:
!
      INTEGER*2 IARRAY(*),IBIT
!
! IARRAY - Variable in which the flag bits are located.
!          May or may not be an array in the calling program.
! IBIT - Index of bit to test. Bits are numbered starting with
!        1 as the lowest order bit in IARRAY(1), 16 is the
!        sign bit in IARRAY(1), 17 is the lowest order bit
!        in IARRAY(2), etc.
!
! 2.3 OUTPUT Variables:
!
      LOGICAL*2 KBIT, K
      LOGICAL*2 HTEST
!
! KBIT - TRUE if the indicated bit is 1; FALSE if bit is 0.
!
! 2.4 COMMON BLOCKS USED
!
! 2.5 SUBROUTINE INTERFACE
!
!       CALLING SUBROUTINES:
!       CALLED SUBROUTINES:
!
! 3.  LOCAL VARIABLES
!
      INTEGER*2 IA,IB,IC
!
! IA - Array index (which word in array)
! IB - Bit index (which bit in word)
!
! 4.  HISTORY
!   WHO   WHEN   WHAT
!   :16:10:2002: JWR  Introduced IC to eliminate difficulty in converting
!                     the '-i2' version
!   :21:10:2003: KDB  Convert htest to btest 
!   :27.05.2004: jwr  Test on IBIT = 0 added. 
!
! 5.  KBIT PROGRAM STRUCTURE
!
      IF ( IBIT .EQ. 0 ) THEN
           KBIT = .FALSE.
           RETURN
      ENDIF
!
! --- Decompose IBIT into an array index IA and a bit index IB.
!
      IA = (IBIT+15)/16
      IB = IBIT - (IA-1)*16
!
! --- Test the appropriate bit
!
      IC = IB-1
      KBIT = BTEST ( IARRAY(IA), IC )
!
      RETURN
      END  !#!  KBIT   #!#
#endif
