!-----------------------------------------------------------------------
      SUBROUTINE INSTRUMENT(K,KALL,TTOT,TMIN,TMAX)
!$$$  SUBPROGRAM DOCUMENTATION  BLOCK
!                .      .    .                                       .
! SUBPROGRAM:  INSTRUMENT    MONITOR WALL-CLOCK TIMES, ETC.
!   PRGMMR: IREDELL          ORG: NP23        DATE:1998-07-16
!
! ABSTRACT: THIS SUBPROGRAM IS USEFUL IN INSTRUMENTING A CODE
!   BY MONITORING THE NUMBER OF TIMES EACH GIVEN SECTION
!   OF A PROGRAM IS INVOKED AS WELL AS THE MINIMUM, MAXIMUM
!   AND TOTAL WALL-CLOCK TIME SPENT IN THE GIVEN SECTION.
!
! PROGRAM HISTORY LOG:
!   1998-07-16  IREDELL
!
! USAGE:    CALL INSTRUMENT(K,KALL,TTOT,TMIN,TMAX)
!   INPUT ARGUMENT LIST:
!     K        - INTEGER POSITIVE SECTION NUMBER
!                OR MAXIMUM SECTION NUMBER IN THE FIRST INVOCATION
!                OR ZERO TO RESET ALL WALL-CLOCK STATISTICS
!                OR NEGATIVE SECTION NUMBER TO SKIP MONITORING
!                AND JUST RETURN STATISTICS.
!
!   OUTPUT ARGUMENT LIST:
!     KALL     - INTEGER NUMBER OF TIMES SECTION IS CALLED
!     TTOT     - REAL TOTAL SECONDS SPENT IN SECTION
!     TMIN     - REAL MINIMUM SECONDS SPENT IN SECTION
!     TMAX     - REAL MAXIMUM SECONDS SPENT IN SECTION
!
! SUBPROGRAMS CALLED:
!   W3UTCDAT     RETURN THE UTC DATE AND TIME
!   W3DIFDAT     RETURN A TIME INTERVAL BETWEEN TWO DATES
!
! REMARKS:
!   THIS SUBPROGRAM SHOULD NOT BE INVOKED FROM A MULTITASKING REGION.
!   NORMALLY, TIME SPENT INSIDE THIS SUBPROGRAM IS NOT COUNTED.
!   WALL-CLOCK TIMES ARE KEPT TO THE NEAREST MILLISECOND.
!
!   EXAMPLE.
!     CALL INSTRUMENT(2,KALL,TTOT,TMIN,TMAX)    ! KEEP STATS FOR 2 SUBS
!     DO K=1,N
!       CALL SUB1
!       CALL INSTRUMENT(1,KALL,TTOT,TMIN,TMAX)  ! ACCUM STATS FOR SUB1
!       CALL SUB2
!       CALL INSTRUMENT(2,KALL,TTOT,TMIN,TMAX)  ! ACCUM STATS FOR SUB2
!     ENDDO
!     PRINT *,'SUB2 STATS: ',KALL,TTOT,TMIN,TMAX
!     CALL INSTRUMENT(-1,KALL,TTOT,TMIN,TMAX)   ! RETURN STATS FOR SUB1
!     PRINT *,'SUB1 STATS: ',KALL,TTOT,TMIN,TMAX
!
! ATTRIBUTES:
!   LANGUAGE: FORTRAN 90
!
!$$$
        IMPLICIT NONE
        INTEGER,INTENT(IN):: K
        INTEGER,INTENT(OUT):: KALL
        REAL,INTENT(OUT):: TTOT,TMIN,TMAX
        INTEGER,SAVE:: KMAX=0
        INTEGER,DIMENSION(:),ALLOCATABLE,SAVE:: KALLS
        REAL,DIMENSION(:),ALLOCATABLE,SAVE:: TTOTS,TMINS,TMAXS
        INTEGER,DIMENSION(8),SAVE:: IDAT
        INTEGER,DIMENSION(8):: JDAT
        REAL,DIMENSION(5):: RINC
        INTEGER:: KA
! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
        KA=ABS(K)
!  ALLOCATE MONITORING ARRAYS IF INITIAL INVOCATION
        IF(KMAX.EQ.0) THEN
          KMAX=K
          ALLOCATE(KALLS(KMAX))
          ALLOCATE(TTOTS(KMAX))
          ALLOCATE(TMINS(KMAX))
          ALLOCATE(TMAXS(KMAX))
          KALLS=0
          KA=0
!  OR RESET ALL STATISTICS BACK TO ZERO
        ELSEIF(K.EQ.0) THEN
          KALLS=0
!  OR COUNT TIME SINCE LAST INVOCATION AGAINST THIS SECTION
        ELSEIF(K.GT.0) THEN
          CALL W3UTCDAT(JDAT)
          CALL W3DIFDAT(JDAT,IDAT,4,RINC)
          KALLS(K)=KALLS(K)+1
          IF(KALLS(K).EQ.1) THEN
            TTOTS(K)=RINC(4)
            TMINS(K)=RINC(4)
            TMAXS(K)=RINC(4)
          ELSE
            TTOTS(K)=TTOTS(K)+RINC(4)
            TMINS(K)=MIN(TMINS(K),RINC(4))
            TMAXS(K)=MAX(TMAXS(K),RINC(4))
          ENDIF
        ENDIF
! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
!  RETURN STATISTICS
        IF(KA.GE.1.AND.KA.LE.KMAX.AND.KALLS(KA).GT.0) THEN
          KALL=KALLS(KA)
          TTOT=TTOTS(KA)
          TMIN=TMINS(KA)
          TMAX=TMAXS(KA)
        ELSE
          KALL=0
          TTOT=0
          TMIN=0
          TMAX=0
        ENDIF
! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
!  KEEP CURRENT TIME FOR NEXT INVOCATION
        IF(K.GE.0) CALL W3UTCDAT(IDAT)
      END SUBROUTINE INSTRUMENT
