      FUNCTION   ATT_ITU_R10 ( P, PW, T, F )
! ************************************************************************
! *                                                                      *
! *   Routine ATT_ITU_R10 computes the coefficients of atmospheric       *
! *   attentuation in the moist air in a range [1, 1000] GHz by taking   *
! *   into account water vapor and oxigen. The atmospheric attenuation   *
! *   is computed in accordance with Recommendation ITU-R P.676-10,      *
! *   09/2013, https://www.itu.int/rec/R-REC-P.676/en                    *
! *   NB: effects of liquid and solid water (rain droplets and           *
! *   snowflakes) is not taken into account.                             *
! *                                                                      *
! * ________________________ Input parameters: _________________________ *
! *                                                                      *
! *  P ( REAL*8     ) -- total pressure. Units: Pa.                      *
! * PW ( REAL*8     ) -- partial pressure of water vapor. Units: Pa.     *
! *  T ( REAL*8     ) -- air temperature. Units: K.                      *
! *  F ( REAL*8     ) -- cyclc frequency. Units: Hz.                     *
! *                                                                      *
! * ________________________ Input parameters: _________________________ *
! *                                                                      *
! * <ATT_ITU_R10> ( REAL*8 ) -- attenuatin. Units: 1/m (or Np/m).        *
! *                                                                      *
! *   Copyright (c) 1975-2025 United States Government as represented by *
! *   the Administrator of the National Aeronautics and Space            *
! *   Administration. All Rights Reserved.                               *
! *   License: NASA Open Source Software Agreement (NOSA).               *
! *                                                                      *
! *  ### 10-SEP-2014  ATT_ITU_R10  v1.0 (d)  L. Petrov  10-SEP-2014 ###  *
! *                                                                      *
! ************************************************************************
      IMPLICIT   NONE 
      REAL*8     ATT_ITU_R10
      REAL*8     P, PW, T, F
      REAL*8     CL, CW, CC, CX, F0
      INTEGER*4  MWL, MOL, MW
      PARAMETER  ( MWL = 35 )
      PARAMETER  ( MOL = 44 )
      PARAMETER  ( MW  =  6 )
      REAL*8     FW(MWL), BW(MW,MWL), FO(MOL), AO(MW,MOL)
      INTEGER*4  NN, N1
      DATA       ( FW(NN), (BW(N1,NN), N1=1,MW), NN=1,MWL ) / &
     &             22.235080D9,     0.1130D0,  2.143D0,  28.11D0,  0.69D0,  4.800D0, 1.00D0, & !  1
     &             67.803960D9,     0.0012D0,  8.735D0,  28.58D0,  0.69D0,  4.930D0, 0.82D0, & !  2
     &            119.995940D9,     0.0008D0,  8.356D0,  29.48D0,  0.70D0,  4.780D0, 0.79D0, & !  3
     &            183.310091D9,     2.4200D0,  0.668D0,  30.50D0,  0.64D0,  5.300D0, 0.85D0, & !  4
     &            321.225644D9,     0.0483D0,  6.181D0,  23.03D0,  0.67D0,  4.690D0, 0.54D0, & !  5
     &            325.152919D9,     1.4990D0,  1.540D0,  27.83D0,  0.68D0,  4.850D0, 0.74D0, & !  6
     &            336.222601D9,     0.0011D0,  9.829D0,  26.93D0,  0.69D0,  4.740D0, 0.61D0, & !  7
     &            380.197372D9,    11.5200D0,  1.048D0,  28.73D0,  0.54D0,  5.380D0, 0.89D0, & !  8
     &            390.134508D9,     0.0046D0,  7.350D0,  21.52D0,  0.63D0,  4.810D0, 0.55D0, & !  9
     &            437.346667D9,     0.0650D0,  5.050D0,  18.45D0,  0.60D0,  4.230D0, 0.48D0, & ! 10
     &            439.150812D9,     0.9218D0,  3.596D0,  21.00D0,  0.63D0,  4.290D0, 0.52D0, & ! 11
     &            443.018295D9,     0.1976D0,  5.050D0,  18.60D0,  0.60D0,  4.230D0, 0.50D0, & ! 12
     &            448.001075D9,    10.3200D0,  1.405D0,  26.32D0,  0.66D0,  4.840D0, 0.67D0, & ! 13
     &            470.888947D9,     0.3297D0,  3.599D0,  21.52D0,  0.66D0,  4.570D0, 0.65D0, & ! 14
     &            474.689127D9,     1.2620D0,  2.381D0,  23.55D0,  0.65D0,  4.650D0, 0.64D0, & ! 15
     &            488.491133D9,     0.2520D0,  2.853D0,  26.02D0,  0.69D0,  5.040D0, 0.72D0, & ! 16
     &            503.568532D9,     0.0390D0,  6.733D0,  16.12D0,  0.61D0,  3.980D0, 0.43D0, & ! 17
     &            504.482692D9,     0.0130D0,  6.733D0,  16.12D0,  0.61D0,  4.010D0, 0.45D0, & ! 18
     &            547.676440D9,     9.7010D0,  0.114D0,  26.00D0,  0.70D0,  4.500D0, 1.00D0, & ! 19
     &            552.020960D9,    14.7700D0,  0.114D0,  26.00D0,  0.70D0,  4.500D0, 1.00D0, & ! 20
     &            556.936002D9,   487.4000D0,  0.159D0,  32.10D0,  0.69D0,  4.110D0, 1.00D0, & ! 21
     &            620.700807D9,     5.0120D0,  2.200D0,  24.38D0,  0.71D0,  4.680D0, 0.68D0, & ! 22
     &            645.866155D9,     0.0713D0,  8.580D0,  18.00D0,  0.60D0,  4.000D0, 0.50D0, & ! 23
     &            658.005280D9,     0.3022D0,  7.820D0,  32.10D0,  0.69D0,  4.140D0, 1.00D0, & ! 24
     &            752.033227D9,   239.6000D0,  0.396D0,  30.60D0,  0.68D0,  4.090D0, 0.84D0, & ! 25
     &            841.053973D9,     0.0140D0,  8.180D0,  15.90D0,  0.33D0,  5.760D0, 0.45D0, & ! 26
     &            859.962313D9,     0.1472D0,  7.989D0,  30.60D0,  0.68D0,  4.090D0, 0.84D0, & ! 27
     &            899.306675D9,     0.0605D0,  7.917D0,  29.85D0,  0.68D0,  4.530D0, 0.90D0, & ! 28
     &            902.616173D9,     0.0426D0,  8.432D0,  28.65D0,  0.70D0,  5.100D0, 0.95D0, & ! 29
     &            906.207325D9,     0.1876D0,  5.111D0,  24.08D0,  0.70D0,  4.700D0, 0.53D0, & ! 30
     &            916.171582D9,     8.3400D0,  1.442D0,  26.70D0,  0.70D0,  4.780D0, 0.78D0, & ! 31
     &            923.118427D9,     0.0869D0, 10.220D0,  29.00D0,  0.70D0,  5.000D0, 0.80D0, & ! 32
     &            970.315022D9,     8.9720D0,  1.920D0,  25.50D0,  0.64D0,  4.940D0, 0.67D0, & ! 33
     &            987.926764D9,   132.1000D0,  0.258D0,  29.85D0,  0.68D0,  4.550D0, 0.90D0, & ! 34
     &           1780.000000D9, 22300.0000D0,  0.952D0, 176.20D0,  0.50D0, 30.500D0, 5.00D0  & ! 35
     &           /
!
       DATA         ( FO(NN), (AO(N1,NN), N1=1,MW), NN=1,MOL ) / &
     &             50.474214D9,    0.975D0, 9.651D0,  6.690D0, 0.0D0,  2.566D0,  6.850D0, & !  1
     &             50.987745D9,    2.529D0, 8.653D0,  7.170D0, 0.0D0,  2.246D0,  6.800D0, & !  2
     &             51.503360D9,    6.193D0, 7.709D0,  7.640D0, 0.0D0,  1.947D0,  6.729D0, & !  3
     &             52.021429D9,   14.320D0, 6.819D0,  8.110D0, 0.0D0,  1.667D0,  6.640D0, & !  4
     &             52.542418D9,   31.240D0, 5.983D0,  8.580D0, 0.0D0,  1.388D0,  6.526D0, & !  5
     &             53.066934D9,   64.290D0, 5.201D0,  9.060D0, 0.0D0,  1.349D0,  6.206D0, & !  6
     &             53.595775D9,  124.600D0, 4.474D0,  9.550D0, 0.0D0,  2.227D0,  5.085D0, & !  7
     &             54.130025D9,  227.300D0, 3.800D0,  9.960D0, 0.0D0,  3.170D0,  3.750D0, & !  8
     &             54.671180D9,  389.700D0, 3.182D0, 10.370D0, 0.0D0,  3.558D0,  2.654D0, & !  9
     &             55.221384D9,  627.100D0, 2.618D0, 10.890D0, 0.0D0,  2.560D0,  2.952D0, & ! 10
     &             55.783815D9,  945.300D0, 2.109D0, 11.340D0, 0.0D0, -1.172D0,  6.135D0, & ! 11
     &             56.264774D9,  543.400D0, 0.014D0, 17.030D0, 0.0D0,  3.525D0, -0.978D0, & ! 12
     &             56.363399D9, 1331.800D0, 1.654D0, 11.890D0, 0.0D0, -2.378D0,  6.547D0, & ! 13
     &             56.968211D9, 1746.600D0, 1.255D0, 12.230D0, 0.0D0, -3.545D0,  6.451D0, & ! 14
     &             57.612486D9, 2120.100D0, 0.910D0, 12.620D0, 0.0D0, -5.416D0,  6.056D0, & ! 15
     &             58.323877D9, 2363.700D0, 0.621D0, 12.950D0, 0.0D0, -1.932D0,  0.436D0, & ! 16
     &             58.446588D9, 1442.100D0, 0.083D0, 14.910D0, 0.0D0,  6.768D0, -1.273D0, & ! 17
     &             59.164204D9, 2379.900D0, 0.387D0, 13.530D0, 0.0D0, -6.561D0,  2.309D0, & ! 18
     &             59.590983D9, 2090.700D0, 0.207D0, 14.080D0, 0.0D0,  6.957D0, -0.776D0, & ! 19
     &             60.306056D9, 2103.400D0, 0.207D0, 14.150D0, 0.0D0, -6.395D0,  0.699D0, & ! 20
     &             60.434778D9, 2438.000D0, 0.386D0, 13.390D0, 0.0D0,  6.342D0, -2.825D0, & ! 21
     &             61.150562D9, 2479.500D0, 0.621D0, 12.920D0, 0.0D0,  1.014D0, -0.584D0, & ! 22
     &             61.800158D9, 2275.900D0, 0.910D0, 12.630D0, 0.0D0,  5.014D0, -6.619D0, & ! 23
     &             62.411220D9, 1915.400D0, 1.255D0, 12.170D0, 0.0D0,  3.029D0, -6.759D0, & ! 24
     &             62.486253D9, 1503.000D0, 0.083D0, 15.130D0, 0.0D0, -4.499D0,  0.844D0, & ! 25
     &             62.997984D9, 1490.200D0, 1.654D0, 11.740D0, 0.0D0,  1.856D0, -6.675D0, & ! 26
     &             63.568526D9, 1078.000D0, 2.108D0, 11.340D0, 0.0D0,  0.658D0, -6.139D0, & ! 27
     &             64.127775D9,  728.700D0, 2.617D0, 10.880D0, 0.0D0, -3.036D0, -2.895D0, & ! 28
     &             64.678910D9,  461.300D0, 3.181D0, 10.380D0, 0.0D0, -3.968D0, -2.590D0, & ! 29
     &             65.224078D9,  274.000D0, 3.800D0,  9.960D0, 0.0D0, -3.528D0, -3.680D0, & ! 30
     &             65.764779D9,  153.000D0, 4.473D0,  9.550D0, 0.0D0, -2.548D0, -5.002D0, & ! 31
     &             66.302096D9,   80.400D0, 5.200D0,  9.060D0, 0.0D0, -1.660D0, -6.091D0, & ! 32
     &             66.836834D9,   39.800D0, 5.982D0,  8.580D0, 0.0D0, -1.680D0, -6.393D0, & ! 33
     &             67.369601D9,   18.560D0, 6.818D0,  8.110D0, 0.0D0, -1.956D0, -6.475D0, & ! 34
     &             67.900868D9,    8.172D0, 7.708D0,  7.640D0, 0.0D0, -2.216D0, -6.545D0, & ! 35
     &             68.431006D9,    3.397D0, 8.652D0,  7.170D0, 0.0D0, -2.492D0, -6.600D0, & ! 36
     &             68.960312D9,    1.334D0, 9.650D0,  6.690D0, 0.0D0, -2.773D0, -6.650D0, & ! 37
     &            118.750334D9,  940.300D0, 0.010D0, 16.640D0, 0.0D0, -0.439D0,  0.079D0, & ! 38
     &            368.498246D9,   67.400D0, 0.048D0, 16.400D0, 0.0D0,  0.000D0,  0.000D0, & ! 39
     &            424.763020D9,  637.700D0, 0.044D0, 16.400D0, 0.0D0,  0.000D0,  0.000D0, & ! 40
     &            487.249273D9,  237.400D0, 0.049D0, 16.000D0, 0.0D0,  0.000D0,  0.000D0, & ! 41
     &            715.392902D9,   98.100D0, 0.145D0, 16.000D0, 0.0D0,  0.000D0,  0.000D0, & ! 42
     &            773.839490D9,  572.300D0, 0.141D0, 16.200D0, 0.0D0,  0.000D0,  0.000D0, & ! 43
     &            834.145546D9,  183.100D0, 0.145D0, 14.700D0, 0.0D0,  0.000D0,  0.000D0  & ! 44
     &           /
!
      INTEGER*4  J1, J2
      REAL*8     TETA, LSTR, LSHA, DF, DELTA, D, WLIN, OLIN, DCNT
      TETA  = 300.0D0/T
!
! --- Comtrubution of water vapor spectral lines
!
      WLIN = 0.0D0
      DO 410 J1=1,MWL-1
!
! ------ Line strength
!
         LSTR = 1.D-3*BW(1,J1) * PW * TETA**3.5D0 * DEXP( BW(2,J1) * (1.0D0 - TETA) )
!
! ------ Line width
!
         DF = 1.D-6*BW(3,J1)*( (P-PW) * TETA**BW(4,J1) + BW(5,J1)*PW * TETA**BW(6,J1) )
!
! ------ Modification of the line width for Doppler broadening
!
         DF = 0.535D0*DF + DSQRT ( 0.217D0*DF**2 + 2.1316D-30 * FW(J1)**2/ TETA )
!
! ------ Line shape
!
         LSHA = F/FW(J1)* ( DF/(1.D-18*(F - FW(J1))**2 + DF**2) + &
     &                      DF/(1.D-18*(F + FW(J1))**2 + DF**2)   &
     &                    )
         WLIN = WLIN + LSTR*LSHA
 410  CONTINUE 
!
! --- Contribution of oxigen spectral lines
!
      OLIN = 0.0D0
      DO 420 J2=1,MOL
!
! ------ Line strength
!
         IF ( F > FO(38) .AND. J2 < 38 ) GOTO 420
         LSTR  = 1.D-9*AO(1,J2) * (P-PW) * TETA**3.0D0 * DEXP( AO(2,J2) * (1.0D0 - TETA) )
!
! ------ Line width
!
         DF    = 1.D-6*AO(3,J2)*( (P-PW) * TETA**(0.8 - AO(4,J2)) + 1.1*PW * TETA)
!
! ------ Modification of the line width for Doppler broadening
!
         DF = DSQRT ( DF**2 + 2.25D-6 )
!
! ------ Correction factor which arises due to interference effects in oxygen lines:
!
         DELTA = 1.D-15*(AO(5,J2) + AO(6,J2)*TETA) * P * TETA**0.8
!
! ------ Line shape
!
         LSHA  = F/FO(J2)* ( (DF - DELTA*(FO(J2) - F))/(1.D-18*(FO(J2) - F)**2 + DF**2) + &
     &                       (DF - DELTA*(FO(J2) + F))/(1.D-18*(FO(J2) + F)**2 + DF**2)   &
     &                     )
         OLIN = OLIN + LSTR*LSHA
 420  CONTINUE 
!
! --- Dry air continuum absorbtion
!
      D = 5.6D-6 * P*TETA**0.8
      DCNT = F * (P-PW)* TETA**2 * &
     &           ( 6.14D-16/(D*(1.0D0 + 1.0D-18*(F/D)**2)) + &
     &                     1.4D-25*(P-PW)* TETA**1.5D0/ &
     &                    (1.0D0 + 6.0D-19*F**1.5D0) &
     &           )
!
! --- Sum the contribution of 
! ---     water spetcral vapor lines,
! ---     oxigen spetcral vapor lines, and
! ---     dry air continuum absorbtin
!
      ATT_ITU_R10 = 4.191D-14 * F * ( WLIN + OLIN + DCNT )
!      
      RETURN
      END  FUNCTION   ATT_ITU_R10  !#!#
