#!/usr/bin/env python3
# ************************************************************************
# *                                                                      *
# *   Routune sched_2hds combines two input VLBI schedules in vex        *
# *   format into an output file that supports two hardware setups.      *
# *                                                                      *
# *   Restrictions:                                                      *
# *                                                                      *
# *   Only vex generated by sur_sked is supported.                       *
# *                                                                      *
# *   Copyright (c) 1975-2025 United States Government as represented by *
# *   the Administrator of the National Aeronautics and Space            *
# *   Administration. All Rights Reserved.                               *
# *   License: NASA Open Source Software Agreement (NOSA).               *
# *                                                                      *
# * ### 09-JAN-2023 sched_2hds.py  v1.0 (d)  L. Petrov  30-JAN-2023  ### *
# *                                                                      *
# ************************************************************************
import pwd, sys, os, shutil, signal, time, subprocess, datetime, argparse
from   sur_sked_config import * # Import sur_sked confuguration
from   pet_misc        import *

sched_2modes__label = "sched_2hds v 1.0 of 2023.12.17"

if ( len(sys.argv)-1 < 3 ):
     print ( "Usage: sched_2modes vex1 vex2 vexout [2/3]" )
     exit  ( 1 )
else:
     vex1_fil = sys.argv[1]
     vex2_fil = sys.argv[2]
     vexo_fil = sys.argv[3]
     if ( len(sys.argv)-1 >= 4 ):
          nsub = int(sys.argv[4])
     else:
          nsub = 2


#
# --- Read the input vex files
#
vex1 = read_file ( vex1_fil )
if ( not vex1 ):
     print ( "Cannot find the first  vex file %s" % vex1_fil )
     exit  ( 1 )

vex2 = read_file ( vex2_fil )
if ( not vex2 ):
     print ( "Cannot find the second vex file %s" % vex2_fil )
     exit  ( 1 )

mode2 = "??"
hds2  = "??"
freq2_buf  = []
mode2_buf  = []
extn2_buf  = []
smod_dic   = {}
fl_freq    = False
fl_bbc     = False
fl_smod    = False
fl_setmod  = False
fl_mode    = False
fl_modedef = False
fl_extn    = False
fl_extndef = False
#
# --- Collect information from the 2nd vex file
#
for line in vex2:
    if ( fl_freq ):
         if ( " def " in line ):
#
# ----------- Learn the mode name
#
              mode2= line.split()[1].replace("_freq;","")
#
# ------ Collect frequency definitions for mode
#
         freq2_buf.append ( line )
         if ( "enddef;" in line ):
              freq2_buf.append("*-------- end   $FREQ")
              fl_freq = False

    if ( line[0:6] == "$FREQ;" ): fl_freq = True

    if ( line[0:6] == "$MODE;" ): fl_mode = True
    if ( fl_mode ):
         if ( line.split()[0] == "def" ):
              fl_modedef = True

    if ( fl_modedef ):
#
# ------ Collect mode definitions for mode2
#
         mode2_buf.append ( line )
         if ( "enddef;" in line ):
              fl_mode    = False
              fl_modedef = False

    if ( line[0:12] == "$EXTENSIONS;" ): fl_extn = True
    if ( fl_extn ):
          if ( " def " in line ):
              fl_extndef = True
    if ( fl_extndef ):
         if ( "def BACKEND_DESCR" in line ):
              fl_extn    = False
              fl_extndef = False
         else:
#
# ----------- Collect the extension block that specifies procedure durations
#
              line = line.replace("__@MOD1@__",mode2)
              if ( not ( "@MOD2@" in line ) and \
                   not ( "@MOD3@" in line ) and \
                   not ( "@MOD4@" in line ) and \
                   not ( "@MOD5@" in line ) and \
                   not ( "@MOD6@" in line ) and \
                   not ( "@MOD7@" in line ) and \
                   not ( "@MOD8@" in line )     ):
                   extn2_buf.append ( line )

    if ( line[0:5] == "$BBC;" ): fl_bbc = True
    if ( fl_bbc ):
         if ( " def " in line ):
#
# ----------- Learn the hardware setup mode name 
#
              hds2= line.split()[1].replace("_bbc;","")
              fl_bbc = False

    if ( "def FS_PROC_DEF_" + hds2 + "_" in line ):
#
# ------ Initilize the item of the procedure definitions for station sta
#
         sta = line.split()[1].replace("FS_PROC_DEF_" + hds2 + "_","").replace(";","")
         smod_dic[sta] = []
         fl_smod = True
    if ( fl_smod ):
         if ( line.split()[0] == "define" and line.split()[1] == "setmode_" + mode2 ):
              fl_setmod = True
    if ( fl_setmod ):
#
# ------ Collect procedure definitions for staiton sta
#
         smod_dic[sta].append ( line )
         if ( "enddef" in line ):
              fl_smod   = False
              fl_setmod = False
              

mode1 = "??"
hds1  = "??"

setmode_dur_max = 0.0
vexo = []

scan_buf   = []
nsca       = 0
fl_freq    = False
fl_smod    = False
fl_freq    = False
fl_scan    = False
fl_mode    = False
fl_modedef = False
fl_extn    = False
fl_extndef = False
#
# --- Process the 1st vex file and put the lines of the combined schedule
# --- into vexo buffer
#
for line in vex1:
    if ( fl_freq ):
         if ( " def " in line ):
#
# ----------- Learn the mode of the 1st schedule
#
              mode1= line.split()[1].replace("_freq;","")

    if ( line[0:6] == "$FREQ;" ): fl_freq = True
    if ( fl_freq ):
         if ( "enddef;" in line ):
#
# ----------- We reached the end of FREQ section.
# ----------- Adde frequency definitions of the 2nd schedule
#
              vexo.append ( line )
              vexo.append ( '*' )
              vexo.append ( '* --- The second frequency setup' )
              vexo.append ( '*' )
              for lin in freq2_buf:
                  vexo.append ( lin )
              fl_freq = False
              continue

    if ( line[0:6] == "$MODE;" ): fl_mode = True
    if ( fl_mode ):
         if ( line.split()[0] == "def" ):
              fl_modedef = True

    if ( fl_modedef ):
         if ( "enddef;" in line ):
#
# ----------- We reached the end of MODE section.
# ----------- Add mode definitions of the 2nd schedule
#
              vexo.append ( line )
              vexo.append ( '*' )
              vexo.append ( '* --- The second mode' )
              vexo.append ( '*' )
              for lin in mode2_buf:
                  vexo.append ( lin.replace("__@MOD1@__",mode2)  )
              fl_mode    = False
              fl_modedef = False
              continue

    if ( line[0:12] == "$EXTENSIONS;" ): fl_extn = True
    if ( fl_extn ):
         if ( " def " in line ):
              fl_extndef = True
    if ( fl_extndef ):
         if ( "def BACKEND_DESCR" in line ):
#
# ----------- We reached the end of EXTENSION settings of the 1st mode.
#
              vexo.append ( '* --- The extension block for the second mode' )
              vexo.append ( '*' )
              for lin in extn2_buf:
                  vexo.append ( lin )
              fl_freq = False

              vexo.append ( line )
              fl_extn = False
              continue

    if ( line[0:5] == "$BBC;" ): fl_bbc = True
    if ( fl_bbc ):
         if ( " def " in line ):
#
# ----------- Learn the hardware mode of the 1st schedule
#
              hds1= line.split()[1].replace("_bbc;","")
              fl_bbc = False

    if ( "extension = NASA" in line ):
         if ( line.split()[6] == "setmode" ):
#
# ----------- Learn the mode setting duration
#
              setmode_dur = float(line.split()[8])
#
# ----------- Update the maximum mode change duration among stations
#
              setmode_dur_max = max ( setmode_dur_max, setmode_dur )

    if ( "def FS_PROC_DEF_" + hds1 + "_" in line ):
         sta = line.split()[1].replace("FS_PROC_DEF_" + hds1 + "_","").replace(";","")
         fl_smod = True
    if ( fl_smod ):
         if ( line.split()[0] == "define" and line.split()[1] == "setmode_" + mode1 ):
              fl_setmod = True
    if ( fl_setmod ):
         if ( "enddef" in line ):
#
# ----------- Added setmode procedure from the 2nd schedule
#
              vexo.append ( line )
              vexo.append ( '        "' )
              vexo.append ( '        "=================================' )
              vexo.append ( '        "' )
              for lin in smod_dic[sta]:
                  vexo.append ( lin )

              fl_setmod = False
              fl_smod   = False
              continue

    if ( line[0:5] == "scan " ):
         fl_scan  = True
         scan_buf = []         

    if ( fl_scan ): 
#
# ------ Processing a scan
#
         line = line.replace("station=","station =" )
#
# ------ Add scan definitions into scan_buf
#
         scan_buf.append ( line )
         if ( line[0:8] == "endscan;" ):
#
# ----------- We reached the end of a given scan in the 1st schedule
#
              nsca = nsca + 1

#
# ----------- Learn scan nominal start and stop time. 
# ----------- Converc this time into pyton fromat
#
              for lin in scan_buf:
                  if ( "start=" in lin ):
                        start_date_str = lin.replace("start=","").replace(";","").ljust(20).strip()
                        start_date= datetime.datetime.strptime ( start_date_str, '%Yy%jd%Hh%Mm%Ss' )
                  if ( "stop=" in lin ):
                        stop_date_str = lin.replace("stop=","").replace(";","").replace("*","").ljust(20).strip()
                        stop_date= datetime.datetime.strptime ( stop_date_str, '%Yy%jd%Hh%Mm%Ss' )
                        time_delta = stop_date - start_date 
                        scan_dur = float(time_delta.seconds)

#
# ----------- Process the 1st subscan
#
              if ( nsub == 2 ):
                   subscan_dur = (scan_dur - setmode_dur_max)/2.0 
              elif ( nsub == 3 ):
                   subscan_dur = (scan_dur - 2*setmode_dur_max)/4.0 
              else:
                   print ( "unsupported 4th argument: 2 or 3 were expected, but got ", sys.argv[4] )
                   exit  (  1 )

              vexo.append ( "*" )
              vexo.append ( "* New scan" )
              vexo.append ( "*" )

              for lin in scan_buf:
                  if ( lin[0:5] == "scan " ):
#
# -------------------- Alter the scan counter
#
                       lin = "scan No%04d;" % nsca
                  if ( "start=" in lin ):
#
# --------------------- Set the start date
#
                        start_date_1st = start_date
                        start_date_str = datetime.datetime.strftime ( start_date_1st, '%Yy%jd%Hh%Mm%Ss;' )
                        lin = "    start=" + start_date_str
                  if ( "stop=" in lin ):
#
# --------------------- Alter the stop date: sustgract change mode time and divide results by 2.
#
                        stop_date_1st = start_date + datetime.timedelta ( seconds=subscan_dur )
                        stop_date_str = datetime.datetime.strftime ( stop_date_1st, '%Yy%jd%Hh%Mm%Ss;' )
                        lin = "*    stop=" + stop_date_str
                  if ( "station =" in lin ):
#
# --------------------- Alter scan duration. FIXME: need alter amount of recorded bites
#
                        old_dur_str = "%4s" % lin.split()[6]
                        dur_str = "%4d" % int(subscan_dur)
                        lin = lin.replace(old_dur_str+" sec",dur_str+" sec")
                  vexo.append ( lin )

              vexo.append ( "*" )
              vexo.append ( "* 2nd subscan" )
              vexo.append ( "*" )
              if ( nsub == 2 ):
                   subscan2_dur =   subscan_dur
              elif ( nsub == 3 ):
                   subscan2_dur = 2*subscan_dur
         
#
# ----------- Process the 2nd subscan
#
              nsca = nsca + 1
              for lin in scan_buf:
                  if ( lin[0:5] == "scan " ):
#
# -------------------- Alter the scan counter
#
                       lin = "scan No%04d;" % nsca
                  if ( "mode=" in lin ):
#
# -------------------- Alter mode name
#
                       lin = lin.replace(mode1,mode2)
                  if ( "start=" in lin ):
#
# --------------------- Set start date
#
                        start_date_2nd = stop_date_1st + datetime.timedelta ( seconds=setmode_dur_max )
                        start_date_str = datetime.datetime.strftime ( start_date_2nd, '%Yy%jd%Hh%Mm%Ss;' )
                        lin = "    start=" + start_date_str
                  if ( "stop=" in lin ):
#
# --------------------- Set the stop date
#
                        stop_date_2nd = start_date_2nd + datetime.timedelta ( seconds=subscan2_dur )
                        stop_date_str = datetime.datetime.strftime ( stop_date_2nd, '%Yy%jd%Hh%Mm%Ss;' )
                        lin = "*    stop=" + stop_date_str
                  if ( "station =" in lin ):
#
# --------------------- Set scan duration. FIXME: need alter amount of recorded bites
#
                        old_dur_str = "%4s" % lin.split()[6]
                        dur_str = "%4d" % int(subscan2_dur)
                        lin = lin.replace(old_dur_str+" sec",dur_str+" sec")
                  vexo.append ( lin )
            
              if ( nsub == 3 ):
                   vexo.append ( "*" )
                   vexo.append ( "* 3rd subscan" )
                   vexo.append ( "*" )
#
# ---------------- Process the 2nd subscan
#
                   nsca = nsca + 1
                   for lin in scan_buf:
                       if ( lin[0:5] == "scan " ):
#
# ------------------------- Alter the scan counter
#
                            lin = "scan No%04d;" % nsca

                       if ( "mode=" in lin ):
#
# ------------------------- Alter mode name
#
                            lin = lin.replace(mode2,mode1)

                       if ( "start=" in lin ):
#
# -------------------------- Set start date
#
                             start_date_3rd = stop_date_2nd + datetime.timedelta ( seconds=setmode_dur_max )
                             start_date_str = datetime.datetime.strftime ( start_date_3rd, '%Yy%jd%Hh%Mm%Ss;' )
                             lin = "    start=" + start_date_str

                       if ( "stop=" in lin ):
#
# -------------------------- Set the stop date
#
                             stop_date_3rd = start_date_3rd + datetime.timedelta ( seconds=subscan_dur )
                             stop_date_str = datetime.datetime.strftime ( stop_date_3rd, '%Yy%jd%Hh%Mm%Ss;' )
                             lin = "*    stop=" + stop_date_str

                       if ( "station =" in lin ):
#
# -------------------------- Set scan duration. FIXME: need alter amount of recorded bites
#
                             old_dur_str = "%4s" % lin.split()[6]
                             dur_str = "%4d" % int(subscan_dur)
                             lin = lin.replace(old_dur_str+" sec",dur_str+" sec")
                       vexo.append ( lin )

              fl_scan = False
         continue
         
    vexo.append ( line )
    if ( line[0:8] == "* proto:" ):
         line = "* Original vex file was processed with " + sched_2modes__label
         vexo.append ( line )

#
# --- Write the output VLBI schdule in VEX format
#
(ret,out) = write_file ( vexo, vexo_fil )
check_err_exe ( ret, out, "write_file" )

print ( "Output file: ", vexo_fil )
