      SUBROUTINE PARSE_HAR_LISTING ( NBUF, BUF, IND_SEC, L_HSP, HSP, IUER )
! ************************************************************************
! *                                                                      *
! *   Routine PARSE_HAR_LISTING parses listing of solution in SPOOL      *
! *   format generated by Solve. It extracts information related to      *
! *   estimation of harmonics site position variations and puts it in    *
! *   arrays of HSP objects: site names, name of the harmonics, phase    *
! *   and frequency of the harmonics, the estimate of sine and cosine    *
! *   amplitudes of X, Y and Z component of displacements and the        *
! *   covariance matrix between estimates of these amplitdes of          *
! *   different stations of the same harmonics.                          *
! *                                                                      *
! * _________________________ Input parameters: ________________________ *
! *                                                                      *
! *    NBUF ( INTEGER*4 ) -- The number of lines in the buffer which     *
! *                          keeps the spool file or the portion of      *
! *                          spool-file with relevant information about  *
! *                          estimates of B-spline.                      *
! *     BUF ( CHARACTER ) -- Buffer with contents of the Solve listing   *
! *                          in SPOOL format. Dimenstion: NBUF.          *
! * IND_SEC ( INTEGER*4 ) -- The index of the first line of the buffer   *
! *                          with respect to the beginning of the        *
! *                          listing. If tyhe buffer contains entire     *
! *                          listing IND_SEC should be 1. This parameter *
! *                          is used for formatting errors only.         *
! *   L_HSP ( INTEGER*4 ) -- The number of harmonics for which position  *
! *                          variations were estimated.                  *
! *                                                                      *
! * ________________________ Modified parameters: ______________________ *
! *                                                                      *
! *     HSP ( RECORD    ) -- Array of objects which keep information     *
! *                          about estimates of harmonics site position  *
! *                          variations and their covariance matrix.     *
! *    IUER ( INTEGER*4, OPT ) -- Universal error handler.               *
! *                           Input: switch IUER=0 -- no error messages  *
! *                                  will be generated even in the case  *
! *                                  of error. IUER=-1 -- in the case of *
! *                                  error the message will be put on    *
! *                                  stdout.                             *
! *                           Output: 0 in the case of successful        *
! *                                   completion and non-zero in the     *
! *                                   case of error.                     *
! *                                                                      *
! *   Copyright (c) 1975-2025 United States Government as represented by *
! *   the Administrator of the National Aeronautics and Space            *
! *   Administration. All Rights Reserved.                               *
! *   License: NASA Open Source Software Agreement (NOSA).               *
! *                                                                      *
! * ### 23-MAR-2005 PARSE_HAR_LISTING v1.0 (d) L. Petrov 23-MAR-2005 ### *
! *                                                                      *
! ************************************************************************
      IMPLICIT   NONE 
      INCLUDE   'astro_constants.i'
      INCLUDE   'solve.i'
      INCLUDE   'hsp.i'
      INTEGER*4  NBUF, IND_SEC, L_HSP, IUER
      CHARACTER  BUF(NBUF)*(*)
      TYPE     ( HSP__TYPE ) :: HSP(L_HSP)
      CHARACTER  STR*32, STR1*32, C_CMP(3)*1, C_CON(2)*1
      DATA       C_CMP / 'X', 'Y', 'Z' /
      DATA       C_CON / 'C', 'S'      /
      REAL*8     VAL
      INTEGER*4  IND_STA
      INTEGER*4  J1, J2, J3, J4, IHSP, K_HSP, L_STA, ICMP, ICON, IOS, &
     &           K_STA, IND_PAR(3*2*MAX_STA), PAR_IND1, PAR_IND2, &
     &           IND1, IND2, IND_COV, L_COV, IER
      INTEGER*4, EXTERNAL :: ILEN, I_LEN, IFIND_PL, LTM_DIF
      INTEGER*4  LOCS, I, J
      LOCS(I,J)=min(I,J) +(max(I,J)*(max(I,J)-1))/2
!
      IHSP = 0
      DO 410 J1=1,NBUF
         CALL CLRCH ( STR1 )
         CALL INCH  ( IND_SEC + J1 - 1, STR1 )
         IF ( BUF(J1)(1:5) == 'L_HPE' ) THEN
              READ ( UNIT=BUF(J1)(7:10), FMT='(I4)', IOSTAT=IOS ) K_HSP
              IF ( IOS .NE. 0 ) THEN
                   CALL CLRCH ( STR ) 
                   CALL INCH  ( IOS, STR ) 
                   CALL ERR_LOG ( 1711, IUER, 'PARSE_HAR_LISTING', 'Error '// &
     &                  STR1(1:I_LEN(STR1))//' in parsing line '//STR )
                   RETURN 
              END IF
              IF ( K_HSP .NE. L_HSP ) THEN
                   CALL ERR_LOG ( 1712, IUER, 'PARSE_HAR_LISTING', 'Trap '// &
     &                 ' of internal control: error in line '// &
     &                 STR1(1:I_LEN(STR1))//' -- '//BUF(J1) )
                   RETURN 
              END IF
            ELSE IF ( BUF(J1)(1:8) == '   I_HPE' ) THEN
              READ ( UNIT=BUF(J1)(10:13), FMT='(I4)', IOSTAT=IOS ) IHSP
              IF ( IOS .NE. 0 ) THEN
                   CALL CLRCH ( STR ) 
                   CALL INCH  ( IOS, STR ) 
                   CALL ERR_LOG ( 1713, IUER, 'PARSE_HAR_LISTING', 'Error '// &
     &                  STR1(1:I_LEN(STR1))//' in parsing line '//STR )
                   RETURN 
              END IF
              IF ( IHSP < 1  .OR.  IHSP > L_HSP ) THEN
                   CALL ERR_LOG ( 1714, IUER, 'PARSE_HAR_LISTING', 'Error '// &
     &                 ' in parsing line '//STR1(1:I_LEN(STR1))//' -- '// &
     &                 BUF(J1) )
                   RETURN 
              END IF
            ELSE IF ( BUF(J1)(1:8) == '   I_STA' ) THEN
              READ ( UNIT=BUF(J1)(10:13), FMT='(I4)', IOSTAT=IOS ) L_STA
              IF ( IOS .NE. 0 ) THEN
                   CALL CLRCH ( STR ) 
                   CALL INCH  ( IOS, STR ) 
                   CALL ERR_LOG ( 1715, IUER, 'PARSE_HAR_LISTING', 'Error '// &
     &                  STR1(1:I_LEN(STR1))//' in parsing line '//STR )
                   RETURN 
              END IF
              IF ( L_STA < 1  .OR.  L_STA > MAX_STA ) THEN
                   CALL ERR_LOG ( 1716, IUER, 'PARSE_HAR_LISTING', 'Error '// &
     &                 ' in parsing line '//STR1(1:I_LEN(STR1))//' -- '// &
     &                 BUF(J1) )
                   RETURN 
              END IF
!
              K_STA = 0
              HSP(IHSP)%L_STA = L_STA
              HSP(IHSP)%L_COV = (6*L_STA*(6*L_STA+1))/2
!
              ALLOCATE     ( HSP(IHSP)%COO(3,L_STA) )
              IF ( IOS .NE. 0 ) THEN
                   CALL CLRCH ( STR )
                   CALL IINCH ( 8*3*L_STA, STR ) 
                   CALL ERR_LOG ( 1717, IUER, 'PARSE_HAR_LISTING', 'Failure '// &
     &                 ' in an attempt to allocate '//STR(1:I_LEN(STR))// &
     &                 ' bytes of dynamic memor fo HSP%COO' )
                   RETURN 
              END IF
!
              ALLOCATE     ( HSP(IHSP)%EST(3,2,L_STA), STAT=IOS )
              IF ( IOS .NE. 0 ) THEN
                   CALL CLRCH ( STR )
                   CALL IINCH ( 8*3*2*L_STA, STR ) 
                   CALL ERR_LOG ( 1718, IUER, 'PARSE_HAR_LISTING', 'Failure '// &
     &                 ' in an attempt to allocate '//STR(1:I_LEN(STR))// &
     &                 ' bytes of dynamic memor fo HSP%EST' )
                   RETURN 
              END IF
!
              ALLOCATE     ( HSP(IHSP)%SIG(3,2,L_STA), STAT=IOS )
              IF ( IOS .NE. 0 ) THEN
                   CALL CLRCH ( STR )
                   CALL IINCH ( 8*3*2*L_STA, STR ) 
                   CALL ERR_LOG ( 1719, IUER, 'PARSE_HAR_LISTING', 'Failure '// &
     &                 ' in an attempt to allocate '//STR(1:I_LEN(STR))// &
     &                 ' bytes of dynamic memor fo HSP%SIG' )
                   RETURN 
              END IF
!
              ALLOCATE     ( HSP(IHSP)%C_STA(L_STA), STAT=IOS )
              IF ( IOS .NE. 0 ) THEN
                   CALL CLRCH ( STR )
                   CALL IINCH ( 8*L_STA, STR ) 
                   CALL ERR_LOG ( 1720, IUER, 'PARSE_HAR_LISTING', 'Failure '// &
     &                 ' in an attempt to allocate '//STR(1:I_LEN(STR))// &
     &                 ' bytes of dynamic memor fo HSP%C_STA' )
                   RETURN 
              END IF
!
              ALLOCATE     ( HSP(IHSP)%COV(HSP(IHSP)%L_COV), STAT=IOS )
              IF ( IOS .NE. 0 ) THEN
                   CALL CLRCH ( STR )
                   CALL IINCH ( 8*L_COV, STR ) 
                   CALL ERR_LOG ( 1721, IUER, 'PARSE_HAR_LISTING', 'Failure '// &
     &                 ' in an attempt to allocate '//STR(1:I_LEN(STR))// &
     &                 ' bytes of dynamic memor fo HSP%COV' )
                   RETURN 
              END IF
!
              ALLOCATE ( HSP(IHSP)%STA(L_STA), STAT=IOS )
              IF ( IOS .NE. 0 ) THEN
                   CALL CLRCH ( STR )
                   CALL IINCH ( L_STA*SIZEOF(HSP(IHSP)%STA(1)), STR ) 
                   CALL ERR_LOG ( 1722, IUER, 'PARSE_HAR_LISTING', &
     &                 'Failure  in an attempt to allocate '// &
     &                  STR(1:I_LEN(STR))//' bytes of dynamic '// &
     &                 'memory for HSP%STA' )
                   RETURN 
              END IF
!
              CALL NOUT_R8 ( 3*2*L_STA, HSP(IHSP)%EST   )
              CALL NOUT_R8 ( 3*2*L_STA, HSP(IHSP)%SIG   )
              CALL NOUT    ( 8*L_STA,   HSP(IHSP)%C_STA )
              CALL NOUT_I4 ( 6*L_STA,   IND_PAR )
              CALL NOUT_R8 ( HSP(IHSP)%L_COV, HSP(IHSP)%COV )
            ELSE IF ( BUF(J1)(1:8) == '   APS:' ) THEN
              IND_STA = LTM_DIF ( 1, K_STA, HSP(IHSP)%C_STA, BUF(J1)(9:16) )
              IF ( IND_STA .LE. 0 ) THEN
                   K_STA = K_STA + 1
                   IND_STA = K_STA
!
                   HSP(IHSP)%C_STA(IND_STA) = BUF(J1)(9:16) 
                   HSP(IHSP)%STA(IND_STA)%FL_EST = .TRUE.
                   HSP(IHSP)%STA(IND_STA)%NAME   = BUF(J1)(9:16) 
                   CALL NOUT_I4 ( 3*2, HSP(IHSP)%STA(IND_STA)%IND_EQU )
              END IF
              ICMP = LTM_DIF ( 1, 3, C_CMP, BUF(J1)(18:18) )
              IF ( ICMP < 1 .OR. ICMP > 3 ) THEN
                   CALL ERR_LOG ( 1723, IUER, 'PARSE_HAR_LISTING', 'Wrong '// &
     &                 'component code. Error in parsing line '// &
     &                 STR1(1:I_LEN(STR1))//' -- '//BUF(J1) )
                   RETURN 
              END IF
!
              READ ( UNIT=BUF(J1)(24:37), FMT='(F14.4)', IOSTAT=IOS ) &
     &               HSP(IHSP)%COO(ICMP,IND_STA)
              IF ( IOS .NE. 0 ) THEN
                   CALL ERR_LOG ( 1724, IUER, 'PARSE_HAR_LISTING', 'Error '// &
     &                 'in format transformation. Error in parsing line '// &
     &                 STR1(1:I_LEN(STR1))//' -- '//BUF(J1) )
                   RETURN 
              END IF
            ELSE IF ( BUF(J1)(1:8) == '   EST:' ) THEN
              IND_STA = LTM_DIF ( 1, K_STA, HSP(IHSP)%C_STA, BUF(J1)(9:16) )
              IF ( IND_STA .LE. 0 ) THEN
                   CALL ERR_LOG ( 1725, IUER, 'PARSE_HAR_LISTING', 'Trap '// &
     &                 'of internal control. Error in parsing line '// &
     &                 STR1(1:I_LEN(STR1))//' -- '//BUF(J1) )
                   RETURN 
              END IF
!
              ICMP = LTM_DIF ( 1, 3, C_CMP, BUF(J1)(18:18) )
              IF ( ICMP < 1 .OR. ICMP > 3 ) THEN
                   CALL ERR_LOG ( 1726, IUER, 'PARSE_HAR_LISTING', 'Wrong '// &
     &                 'component code. Error in parsing line '// &
     &                 STR1(1:I_LEN(STR1))//' -- '//BUF(J1) )
                   RETURN 
              END IF
              ICON = LTM_DIF ( 1, 2, C_CON, BUF(J1)(19:19) )
              IF ( ICON < 1 .OR. ICON > 2 ) THEN
                   CALL ERR_LOG ( 1727, IUER, 'PARSE_HAR_LISTING', 'Wrong '// &
     &                 'constituent code. Error in parsing line '// &
     &                 STR1(1:I_LEN(STR1))//' -- '//BUF(J1) )
                   RETURN 
              END IF
              HSP(IHSP)%HAR_NAME = BUF(J1)(21:28)
              READ ( UNIT=BUF(J1)(33:37), FMT='(I9)', IOSTAT=IOS ) &
     &               HSP(IHSP)%STA(IND_STA)%IND_EQU(ICMP,ICON)
              IF ( IOS .NE. 0 ) THEN
                   CALL ERR_LOG ( 1728, IUER, 'PARSE_HAR_LISTING', 'Error '// &
     &                 'in format transformation. Error in parsing line '// &
     &                 STR1(1:I_LEN(STR1))//' -- '//BUF(J1) )
                   RETURN 
              END IF
              IND_PAR( 6*(IND_STA-1) + 3*(ICON-1) + ICMP ) = HSP(IHSP)%STA(IND_STA)%IND_EQU(ICMP,ICON)
!
              READ ( UNIT=BUF(J1)(42:51), FMT='(F10.5)', IOSTAT=IOS ) &
     &               HSP(IHSP)%STA(IND_STA)%PHASE
              IF ( IOS .NE. 0 ) THEN
                   CALL ERR_LOG ( 1729, IUER, 'PARSE_HAR_LISTING', 'Error '// &
     &                 'in format transformation. Error in parsing line '// &
     &                 STR1(1:I_LEN(STR1))//' -- '//BUF(J1) )
                   RETURN 
              END IF
!
              READ ( UNIT=BUF(J1)(56:73), FMT='(F18.6)', IOSTAT=IOS ) &
     &               HSP(IHSP)%STA(IND_STA)%FREQ
              IF ( IOS .NE. 0 ) THEN
                   CALL ERR_LOG ( 1730, IUER, 'PARSE_HAR_LISTING', 'Error '// &
     &                 'in format transformation. Error in parsing line '// &
     &                 STR1(1:I_LEN(STR1))//' -- '//BUF(J1) )
                   RETURN 
              END IF
!
              READ ( UNIT=BUF(J1)(78:92), FMT='(F15.5)', IOSTAT=IOS ) &
     &               HSP(IHSP)%EST(ICMP,ICON,IND_STA)
              IF ( IOS .NE. 0 ) THEN
                   CALL ERR_LOG ( 1731, IUER, 'PARSE_HAR_LISTING', 'Error '// &
     &                 'in format transformation. Error in parsing line '// &
     &                 STR1(1:I_LEN(STR1))//' -- '//BUF(J1) )
                   RETURN 
              END IF
            ELSE IF ( BUF(J1)(1:8) == '   COV:' ) THEN
              READ ( UNIT=BUF(J1)(15:19), FMT='(I5)', IOSTAT=IOS ) PAR_IND1
              IF ( IOS .NE. 0 ) THEN
                   CALL ERR_LOG ( 1732, IUER, 'PARSE_HAR_LISTING', 'Error '// &
     &                 'in format transformation. Error in parsing line '// &
     &                  STR1(1:I_LEN(STR1))//' -- '//BUF(J1) )
                   RETURN 
              END IF
              IND1 = IFIND_PL ( 6*L_STA, IND_PAR, PAR_IND1 )
              IF ( IND1 .LE. 0 ) THEN
                   CALL ERR_LOG ( 1733, IUER, 'PARSE_HAR_LISTING', 'Trap '// &
     &                 'of internal control. Error in parsing line '// &
     &                  STR1(1:I_LEN(STR1))//' -- '//BUF(J1) )
                   RETURN 
              END IF
!
              READ ( UNIT=BUF(J1)(27:31), FMT='(I5)', IOSTAT=IOS ) PAR_IND2
              IF ( IOS .NE. 0 ) THEN
                   CALL ERR_LOG ( 1734, IUER, 'PARSE_HAR_LISTING', 'Error '// &
     &                 'in format transformation. Error in parsing line '// &
     &                  STR1(1:I_LEN(STR1))//' -- '//BUF(J1) )
                   RETURN 
              END IF
              IND2 = IFIND_PL ( 6*L_STA, IND_PAR, PAR_IND2 )
              IF ( IND2 .LE. 0 ) THEN
                   CALL ERR_LOG ( 1735, IUER, 'PARSE_HAR_LISTING', 'Trap '// &
     &                 'of internal control. Error in parsing line '// &
     &                  STR1(1:I_LEN(STR1))//' -- '//BUF(J1) )
                   RETURN 
              END IF
!
              READ ( UNIT=BUF(J1)(36:54), FMT='(F19.12)', IOSTAT=IOS ) &
     &               HSP(IHSP)%COV( LOCS(IND1,IND2) )
              IF ( IOS .NE. 0 ) THEN
                   CALL ERR_LOG ( 1736, IUER, 'PARSE_HAR_LISTING', 'Error '// &
     &                 'in format transformation. Error in parsing line '// &
     &                  STR1(1:I_LEN(STR1))//' -- '//BUF(J1) )
                   RETURN 
              END IF
            ELSE 
              CALL ERR_LOG ( 1737, IUER, 'PARSE_HAR_LISTING', 'Unsupported '// &
     &            'prefix: '//STR(1:7)//' was fgound while parsing line '// &
     &             STR1(1:I_LEN(STR1))//' -- '//BUF(J1) )
              RETURN 
         END IF
 410  CONTINUE 
!
      CALL ERR_LOG ( 0, IUER )
      RETURN
      END  SUBROUTINE  PARSE_HAR_LISTING 
